% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/esb_test.R
\name{esb.test}
\alias{esb.test}
\title{Perform the Ioannidis test for excess of significance}
\usage{
esb.test(
  x,
  input = "dataframe",
  n_cases = NULL,
  n_controls = NULL,
  measure = NULL,
  method = "binom.test",
  true_effect = "largest",
  seed = NA
)
}
\arguments{
\item{x}{a well-formatted dataset or an object of class \code{rma} or \code{meta}. If a well-formatted dataset is used, only one factor should be included.}

\item{input}{the type of object used as input. It must be \code{dataframe}, \code{rma} or \code{meta}.}

\item{n_cases}{vector with the number of cases of each included studies. Only required when \code{x} is an object of class \code{rma} or \code{meta}}

\item{n_controls}{vector with the number of controls of each included studies. Only required when \code{x} is an object of class \code{rma} or \code{meta}}

\item{measure}{the measure of the effect: "SMD", "OR" or "logOR, "RR" or "logRR", "IRR" or "logIRR", "HR" or "logHR". If a an object of class \dQuote{rma} or \dQuote{meta} is used, the effect size should be either 'SMD' or 'OR'. The measure used can be indicated via the measure argument of the \code{esb.test} function or directly when calling the \code{rma} or \code{meta} functions (see examples below).}

\item{method}{the method used to conduct the test. It must be \code{binom.test} or \code{chisq.test} (see details). By default, the umbrella function uses a \code{binom.test} method.}

\item{true_effect}{the method to estimate the true effect. It must be \code{largest} or a numeric value.}

\item{seed}{an integer value used as an argument by the set.seed() function.}
}
\value{
The dataset contains the following columns: \tabular{ll}{
\code{method} \tab method used to conduct the test.\cr
\tab \cr
\code{p.value} \tab p-value for the test statistic.\cr
\tab \cr
\code{power} \tab the power of each individual study to detect the \dQuote{true_effect} \cr
\tab at an alpha of .05.\cr
\tab \cr
\code{mean_power} \tab the mean power of all individual studies to detect the \dQuote{true_effect}.\cr
\tab \cr
\code{k} \tab the total number of studies.\cr
\tab \cr
\code{sig} \tab whether each individual study has statistically significant results.\cr
\tab \cr
\code{O} \tab the total number of studies with statistically significant results.\cr
\tab \cr
\code{E} \tab the total expected number of studies with statistically significant results.\cr
\tab \cr
}
}
\description{
The \code{esb.test()} function performs the "Ioannidis test" to examine the presence of an excess of significance in a given set of studies.
This test aims to determine whether there is an excess in the observed number of studies with statistically significant results given the mean statistical power.
An exact binomial test or a Chi-squared test is used.
}
\details{
The function starts by calculating whether each individual study has significant results (p < .05). Then, it estimates the statistical power of each individual study to detect the 'true' effect size.
The \code{true_effect} argument can be used to select the method that will be applied to estimate the true effect.
\itemize{
\item If \code{largest} is selected, the true effect size is assumed to be equal to the effect size of the largest study included in the meta-analysis.
\item If a \code{numeric} value is entered, the true effect size is assumed to be equal to the value entered by the user (note that the value of the ratios (OR/RR/IRR/HR) must be in its natural scale).
}

Last, this function performs a statistical test to determine whether the observed number of statistically significant studies is higher than expected given the mean statistical power. The \code{method} argument can be used to select the test.
\itemize{
\item If \code{binom.test} is selected, the function performs a binomial exact test of a simple null hypothesis about the probability of success. In this test, the studies with statistically significant results are considered as successes. The mean statistical power to detect the effect size obtained by the largest study is considered as the probability of success. The exact test is significant if the number of statistically significant studies is different from what could be have been expected given the mean observed power.
\item If \code{chisq.test} is selected, the function performs a chi-square test based on the number of studies with significant results, on the number of studies with non-significant results and their associated probability of occurrence (i.e., their statistical power to detect the effect size obtained by the largest study). The chi-square test is significant if the number of statistically significant studies is different from what could be have been expected given their observed power.
}
}
\examples{
### load a well-formatted dataframe with a single factor
df <- df.SMD[df.SMD$factor == "Surgical", ]

### perform an excess significance bias directly on this dataframe
esb.test(df, measure = "SMD", input = "dataframe")

### perform an excess significance bias on a rma object
rma <- metafor::rma(yi = value, sei = (ci_up - ci_lo) / 3.92,
                    measure = "SMD",
                    data = df)

esb.test(rma, n_cases = df$n_cases, n_controls = df$n_controls, input = "rma")

### perform an excess significance bias on a meta object
meta <- meta::metagen(TE = value, seTE = (ci_up - ci_lo) / 3.92,
                      sm = "SMD",
                      data = df)

esb.test(meta, n_cases = df$n_cases, n_controls = df$n_controls, input = "meta")

### perform an excess significance bias using the umbrella function
umbrella(df)[[1]]$esb
}
\references{
Ioannidis, JPA., Munafo, MR., Fusar-Poli, P., Nosek, BA., & David, SP. (2014). Publication and other reporting biases in cognitive sciences: detection, prevalence, and prevention. \emph{Trends in Cognitive Sciences}, \bold{18}, 235-241.
}
