% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Wdiag.R
\name{thselect.wseq}
\alias{thselect.wseq}
\title{Wadsworth's sequential analysis threshold selection}
\usage{
thselect.wseq(
  xdat,
  thresh,
  qlev,
  model = c("nhpp", "taildep", "rtaildep"),
  npp = 1,
  nsim = 1000L,
  level = 0.95,
  plot = FALSE,
  ...
)
}
\arguments{
\item{xdat}{a numeric vector or matrix of data to be fitted.}

\item{thresh}{vector of candidate thresholds.}

\item{qlev}{vector of probabilities for empirical quantiles used in place of the threshold, used if argument \code{thresh} is missing.}

\item{model}{string specifying whether the univariate or multivariate diagnostic should be used. Either \code{nhpp} for the univariate model, or \code{exp} (\code{invexp}) for the bivariate exponential model with rate (inverse rate) parametrization. See details.}

\item{npp}{number of observations per period for the non-homogeneous point process model. Default to 1.}

\item{nsim}{number of Monte Carlo simulations used to assess the null distribution of the test statistic}

\item{level}{confidence level of intervals, defaults to 0.95}

\item{plot}{logical; if \code{TRUE}, calls the plot routine}

\item{...}{additional parameters passed to internal routine}
}
\value{
an object of class invisible list with components
\itemize{
\item \code{thresh0}: threshold selected by the likelihood ratio procedure
\item \code{thresh}: vector of candidate thresholds
\item \code{coef}: maximum likelihood estimates from all thresholds
\item \code{vcov}: joint asymptotic covariance matrix for shape \eqn{\xi} or coefficient of tail dependence \eqn{\eta}, or it's reciprocal.
\item \code{wn}: values of the white noise process
\item \code{stat}: value of the likelihood ratio test statistic for the changepoint test
\item \code{pval}: \emph{P}-value of the likelihood ratio test
\item \code{mle}: maximum likelihood estimates for the selected threshold
\item \code{model}: model fitted, either \code{nhpp}, \code{exp} or \code{invexp}
\item \code{nsim}: number of Monte Carlo simulations for changepoint test
\item \code{xdat}: vector of observations
}
}
\description{
Function to produce diagnostic plots and test statistics for the
threshold diagnostics exploiting structure of maximum likelihood estimators
based on the non-homogeneous Poisson process likelihood or the coefficient of tail dependence
}
\details{
The function is a wrapper for the univariate (non-homogeneous Poisson process model) and exponential dependence model applied to the minimum component (tail dependence coefficient).
For the latter, the user can select either the rate (\code{"taildep"} or inverse rate parameter  (\code{"rtaildep"}). The inverse rate parametrization  works better for uniformity of the p-value distribution under the likelihood ratio test for the changepoint.

For the coefficient of tail dependence, users must provide pairwise minimum of marginally exponentially distributed margins (see example)
}
\examples{
\dontrun{
set.seed(123)
xdat <- abs(rnorm(5000))
thresh <- quantile(xdat, seq(0, 0.9, by = 0.1))
(diag <- thselect.wseq(
 xdat = xdat,
 thresh = thresh,
 plot = TRUE,
 type = "ps"))
# Multivariate example, with coefficient of tail dependence
xbvn <- rmnorm(n = 6000L,
                mu = rep(0, 2),
                Sigma = cbind(c(1, 0.7), c(0.7, 1)))
thselect.wseq(
  xdat = xbvn,
  qlev = seq(0, 0.9, length.out = 30),
  model = 'taildep',
  plot = TRUE)
}
}
\references{
Wadsworth, J.L. (2016). Exploiting Structure of Maximum Likelihood Estimators for Extreme Value Threshold Selection, \emph{Technometrics}, \bold{58}(1), 116-126, \code{http://dx.doi.org/10.1080/00401706.2014.998345}.
}
\author{
Jennifer L. Wadsworth, Léo Belzile
}
