#' opensensmapr: Get sensor data from opensensemap.org
#'
#' The opensensmapr package provides functions for
#' \itemize{
#'   \item retrieval of senseBox metadata,
#'   \item retrieval of senseBox measurements,
#'   \item general statistics about the openSenseMap database.
#' }
#' Additionally, helper functions are provided to ease the integration with the
#' \code{\link[sf]{sf}} package for spatial analysis as well as
#' \code{\link[dplyr]{dplyr}} for general data handling.
#'
#' @section Retrieving senseBox metadata:
#' On the openSenseMap, measurements are provided by sensors which are assigned
#' to a sensor station ("senseBox").
#' A senseBox consists of a collection of sensors, a location (-history), an ID,
#' as well as metadata about its owner & placement.
#' senseBoxes can be retrieved either by ID, or as a collection with optional
#' filters on their metadata
#' \itemize{
#'   \item \code{\link{osem_box}}: Get metadata about a single box by its ID.
#'   \item \code{\link{osem_boxes}}: Get metadata about all boxes, optionally
#'     filtered by their attributes.
#' }
#'
#' The data is returned as a \code{\link{data.frame}} with the class
#' \code{sensebox} attached.
#' To help in getting an overview of the dataset additional functions are
#' implemented:
#' \itemize{
#'   \item \code{summary.sensebox()}: Aggregate the metadata about the given
#'     list of senseBoxes.
#'   \item \code{plot.sensebox()}: Shows the spatial distribution of the given
#'     list of senseBoxes on a map. Requires additional packages!
#'   \item \code{\link{osem_phenomena}}: Get a named list with
#'     counts of the measured phenomena of the given list of senseBoxes.
#' }
#'
#' @section Retrieving measurements:
#' There are two ways to retrieve measurements:
#' \itemize{
#'   \item \code{\link{osem_measurements_archive}}:
#'     Downloads measurements for a \emph{single box} from the openSenseMap archive.
#'     This function does not provide realtime data, but is suitable for long time frames.
#'     
#'   \item \code{\link{osem_measurements}}:
#'     This function retrieves (realtime) measurements from the API. It works for a
#'     \emph{single phenomenon} only, but provides various filters to select sensors by
#'     
#'     \itemize{
#'       \item a list of senseBoxes, previously retrieved through
#'         \code{\link{osem_box}} or \code{\link{osem_boxes}}.
#'       \item a geographic bounding box, which can be generated with the
#'         \code{\link[sf]{sf}} package.
#'       \item a time frame
#'       \item a exposure type of the given box
#'     }
#'     
#'     Use this function with caution for long time frames, as the API becomes
#'     quite slow is limited to 10.000 measurements per 30 day interval.
#' }
#'
#' Data is returned as \code{tibble} with the class \code{osem_measurements}.
#'
#' @section Retrieving statistics:
#' Count statistics about the database are provided with \code{\link{osem_counts}}.
#'
#' @section Using a different API instance / endpoint:
#' You can override the functions \code{osem_endpoint} and \code{osem_endpoint_archive}
#' inside the package namespace:
#' 
#' \code{
#'   assignInNamespace("osem_endpoint", function() "http://mynewosem.org", "opensensmapr")
#' }
#' 
#' @section Integration with other packages:
#' The package aims to be compatible with the tidyverse.
#' Helpers are implemented to ease the further usage of the retrieved data:
#'
#' \itemize{
#'   \item \code{\link{osem_as_sensebox}} & \code{\link{osem_as_measurements}}:
#'     Transform a foreign object to a sensebox data.frame or osem_measurements
#'     by attaching the required classes and attributes.
#'   \item \code{\link{st_as_sf.sensebox}} & \code{\link{st_as_sf.osem_measurements}}:
#'     Transform the senseBoxes or measurements into an \code{\link[sf]{sf}}
#'     compatible format for spatial analysis.
#'   \item \code{filter.sensebox()} & \code{mutate.sensebox()}: for use with
#'     \code{\link{dplyr}}.
#' }
#'
#' @seealso Report bugs at \url{https://github.com/sensebox/opensensmapR/issues}
#' @seealso openSenseMap API: \url{https://api.opensensemap.org/}
#' @seealso official openSenseMap API documentation: \url{https://docs.opensensemap.org/}
#' @docType package
#' @name opensensmapr
'_PACKAGE'

#' @importFrom graphics plot legend par
#' @importFrom magrittr %>%
`%>%` = magrittr::`%>%`

# just to make R CMD check happy, due to NSE (dplyr) functions
globalVariables(c(
  'createdAt',
  'lastMeasurement',
  'sensorType',
  'title',
  'unit',
  'value',
  'X_id',
  '.'
))
