% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/elnet.R
\name{elnet}
\alias{elnet}
\alias{adaelnet}
\alias{adaen}
\title{Compute the Least Squares (Adaptive) Elastic Net Regularization Path}
\usage{
elnet(
  x,
  y,
  alpha,
  nlambda = 100,
  lambda_min_ratio,
  lambda,
  penalty_loadings,
  weights,
  intercept = TRUE,
  en_algorithm_opts,
  sparse = FALSE,
  eps = 1e-06,
  standardize = TRUE,
  correction = deprecated(),
  xtest = deprecated(),
  options = deprecated()
)
}
\arguments{
\item{x}{\code{n} by \code{p} matrix of numeric predictors.}

\item{y}{vector of response values of length \code{n}.}

\item{alpha}{elastic net penalty mixing parameter with \eqn{0 \le \alpha \le 1}. \code{alpha = 1} is the LASSO penalty,
and \code{alpha = 0} the Ridge penalty.}

\item{nlambda}{number of penalization levels.}

\item{lambda_min_ratio}{Smallest value of the penalization level as a fraction of the largest level (i.e., the
smallest value for which all coefficients are zero). The default depends on the sample
size relative to the number of variables and \code{alpha}. If more observations than variables
are available, the default is \code{1e-3 * alpha}, otherwise \code{1e-2 * alpha}.}

\item{lambda}{optional user-supplied sequence of penalization levels. If given and not \code{NULL}, \code{nlambda} and
\code{lambda_min_ratio} are ignored.}

\item{penalty_loadings}{a vector of positive penalty loadings (a.k.a. weights) for different penalization of each
coefficient.}

\item{weights}{a vector of positive observation weights.}

\item{intercept}{include an intercept in the model.}

\item{en_algorithm_opts}{options for the EN algorithm. See \link{en_algorithm_options}
for details.}

\item{sparse}{use sparse coefficient vectors.}

\item{eps}{numerical tolerance.}

\item{standardize}{standardize variables to have unit variance. Coefficients are always returned in original scale.}

\item{correction}{defunct. Correction for EN estimates is not supported anymore.}

\item{xtest}{deprecated. Instead, extract coefficients with \code{\link[=coef.pense_fit]{coef.pense_fit()}} and compute predictions manually.}

\item{options}{deprecated. Use \code{en_algorithm_opts} instead.}
}
\value{
a list-like object with the following items
\describe{
\item{\code{lambda}}{the sequence of penalization parameters.}
\item{\code{estimates}}{a list of estimates. Each estimate contains the following information:
\describe{
\item{\code{intercept}}{intercept estimate.}
\item{\code{beta}}{beta (slope) estimate.}
\item{\code{lambda}}{penalization level at which the estimate is computed.}
\item{\code{alpha}}{\emph{alpha} hyper-parameter at which the estimate is computed.}
\item{\code{statuscode}}{if \verb{> 0} the algorithm experienced issues when computing the estimate.}
\item{\code{status}}{optional status message from the algorithm.}
}
}
\item{\code{call}}{the original call.}
\item{\code{predictions}}{if \code{xtest} was given, a matrix of predicted values. Each column corresponds to the
predictions from the estimate at the \code{lambda} value at the same index.}
}
}
\description{
Compute least squares EN estimates for linear regression with optional observation
weights and penalty loadings.
}
\details{
The elastic net estimator for the linear regression model solves
the optimization problem

\deqn{argmin_{\mu, \beta}
  (1/2n) \sum_i w_i (y_i - \mu - x_i' \beta)^2 +
  \lambda \sum_j 0.5 (1 - \alpha) \beta_j^2 + \alpha l_j |\beta_j|  }

with observation weights \eqn{w_i} and penalty loadings \eqn{l_j}.
}
\examples{
# Compute the LS-EN regularization path for Freeny's revenue data
# (see ?freeny)
data(freeny)
x <- as.matrix(freeny[ , 2:5])

regpath <- elnet(x, freeny$y, alpha = 0.75)
plot(regpath)

# Extract the coefficients at a certain penalization level
coef(regpath, lambda = regpath$lambda[5])

# What penalization level leads to good prediction performance?
cv_results <- elnet_cv(x, freeny$y, alpha = 0.75, cv_repl = 10,
                       cv_k = 4, cv_measure = 'tau')
plot(cv_results, se_mult = 1)
plot(cv_results, se_mult = 1, what = 'coef.path')

# Extract the coefficients at the penalization level with
# smallest prediction error ...
coef(cv_results)
# ... or at the penalization level with prediction error
# statistically indistinguishable from the minimum.
coef(cv_results, lambda = 'se')
}
\seealso{
\code{\link[=pense]{pense()}} for an S-estimate of regression with elastic net penalty.

\code{\link[=coef.pense_fit]{coef.pense_fit()}} for extracting coefficient estimates.

\code{\link[=plot.pense_fit]{plot.pense_fit()}} for plotting the regularization path.

Other functions for computing non-robust estimates: 
\code{\link{elnet_cv}()}
}
\concept{functions for computing non-robust estimates}
