% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sun.calc.r
\name{day_night}
\alias{day_night}
\alias{day_night_fast}
\alias{noon_time}
\alias{sunrise_time}
\alias{sunset_time}
\alias{day_length}
\alias{night_length}
\title{Times for sun positions}
\usage{
day_night(date = lubridate::now(tzone = "UTC"),
  tz = lubridate::tz(date), geocode = tibble::tibble(lon = 0, lat =
  51.5, address = "Greenwich"), twilight = "none", unit.out = "hours")

day_night_fast(date, tz, geocode, twilight, unit.out)

noon_time(date = lubridate::today(), tz = lubridate::tz(date),
  geocode = tibble::tibble(lon = 0, lat = 51.5, address = "Greenwich"),
  twilight = "none", unit.out = "datetime")

sunrise_time(date = lubridate::today(), tz = lubridate::tz(date),
  geocode = tibble::tibble(lon = 0, lat = 51.5, address = "Greenwich"),
  twilight = "sunlight", unit.out = "datetime")

sunset_time(date = lubridate::today(), tz = lubridate::tz(date),
  geocode = tibble::tibble(lon = 0, lat = 51.5, address = "Greenwich"),
  twilight = "sunlight", unit.out = "datetime")

day_length(date = lubridate::now(), tz = "UTC",
  geocode = tibble::tibble(lon = 0, lat = 51.5, address = "Greenwich"),
  twilight = "sunlight", unit.out = "hours")

night_length(date = lubridate::now(), tz = "UTC",
  geocode = tibble::tibble(lon = 0, lat = 51.5, address = "Greenwich"),
  twilight = "sunlight", unit.out = "hours")
}
\arguments{
\item{date}{"vector" of POSIXct times or Date objects, any valid TZ is allowed,
default is current date at Greenwich.}

\item{tz}{character vector indicating time zone to be used in output.}

\item{geocode}{data frame with one or more rows and variables lon and lat as
numeric values (degrees). If present, address will be copied to the output.}

\item{twilight}{character string, one of "none", "rim", "refraction",
"sunlight", "civil", "nautical", "astronomical", or a \code{numeric} vector
of length one, or two, giving solar elevation angle(s) in degrees (negative
if below the horizon).}

\item{unit.out}{character string, One of "datetime", "day", "hour", "minute",
or "second".}
}
\value{
A tibble with variables day, tz, twilight.rise, twilight.set,
  longitude, latitude, address, sunrise, noon, sunset, daylength,
  nightlength or the corresponding individual vectors.

\code{noon_time}, \code{sunrise_time} and \code{sunset_time} return a
  vector of POSIXct times

\code{day_length} and \code{night_length} return numeric a vector
  giving the length in hours
}
\description{
Functions for calculating the timing of solar positions, given geographical
coordinates and dates. They can be also used to find the time for an
arbitrary solar elevation between 90 and -90 degrees by supplying "twilight"
angle(s) as argument.
}
\details{
Twilight names are interpreted as follows. "none": solar elevation =
  0 degrees. "rim": upper rim of solar disk at the horizon or solar elevation
  = -0.53 / 2. "refraction": solar elevation = 0 degrees + refraction
  correction. "sunlight": upper rim of solar disk corrected for refraction,
  which is close to the value used by the online NOAA Solar Calculator.
  "civil": -6 degrees, "naval": -12 degrees, and "astronomical": -18 degrees.
  Unit names for output are as follows: "day", "hours", "minutes" and
  "seconds" times for sunrise and sunset are returned as times-of-day since
  midnight expressed in the chosen unit. "date" or "datetime" return the same
  times as datetime objects with TZ set (this is much slower than "hours").
  Day length and night length are returned as numeric values expressed in
  hours when `"datetime"' is passed as argument to \code{unit.out}. If
  twilight is a numeric vector of length two, the element with index 1 is
  used for sunrise and that with index 2 for sunset.
}
\note{
This function is an implementation of Meeus equations as used in NOAAs
  on-line web calculator, which are very precise and valid for a very broad
  range of dates. For sunrise and sunset the times are affected by refraction
  in the atmosphere, which does in turn depend on weather conditions. The
  effect of refraction on the apparent position of the sun is only an
  estimate based on "typical" conditions. The more tangential to the horizon
  is the path of the sun, the larger the effect of refraction is on the times
  of visual occlusion of the sun behind the horizon---i.e. the largest timing
  errors occur at high latitudes. The computation is not defined for
  latitudes 90 and -90 degrees, i.e. at the poles.

  There exists a different R implementation of the same algorithms called
  "AstroCalcPureR" available as function \code{astrocalc4r} in package
  'fishmethods'. Although the equations used are almost all the same, the
  function signatures and which values are returned differ. In particular,
  the present implementation splits the calculation into two separate
  functions, one returning angles at given instants in time, and a separate
  one returning the timing of events for given dates.

  In the current implementation functions \code{sunrise_time},
  \code{noon_time}, \code{sunset_time} and \code{day_length} are wrappers
  on \code{day_night}, so if more than one quantity is needed it is
  preferable to directly call \code{day_night} as it will be faster.

\code{night_length} returns the length of night-time conditions in one
  day (00:00:00 to 23:59:59), rather than the length of the night between two
  consecutive days.
}
\section{Warning}{
 Be aware that R's \code{Date} class does not save time zone
  metadata. This can lead to ambiguities in the current implementation
  based on time instants. The argument passed to \code{date} should be
  of class \code{POSIXct}, in other words an instant in time, from which
  the correct date will be computed based on the \code{tz} argument.
}

\examples{
library(lubridate)
my.geocode <- data.frame(lat = 60, lon = 25)
day_night(ymd("2015-05-30"), geocode = my.geocode)
day_night(ymd("2015-05-30") + days(1:10), geocode = my.geocode, twilight = "civil")
sunrise_time(ymd("2015-05-30"), geocode = my.geocode)
noon_time(ymd("2015-05-30"), geocode = my.geocode)
sunset_time(ymd("2015-05-30"), geocode = my.geocode)
day_length(ymd("2015-05-30"), geocode = my.geocode)
day_length(ymd("2015-05-30"), geocode = my.geocode, unit.out = "day")

}
\references{
The primary source for the algorithm used is the book:
Meeus, J. (1998) Astronomical Algorithms, 2 ed., Willmann-Bell, Richmond,
VA, USA. ISBN 978-0943396613.

A different implementation is available at
\url{https://www.nefsc.noaa.gov/AstroCalc4R/} and in R paclage
\code{\link[fishmethods]{astrocalc4r}}. In 'fishmethods' (= 1.11-0) there
is a bug in function astrocalc4r() that affects sunrise and sunset times.

An interactive web page using the same algorithms is available at
\url{https://www.esrl.noaa.gov/gmd/grad/solcalc/}. There are small
differences in the returned times compared to our function that seem to be
related to the estimation of atmospheric refraction (about 0.1 degrees).
}
\seealso{
\code{\link{sun_angles}}.

Other astronomy related functions: \code{\link{format.solar_time}},
  \code{\link{is.solar_time}},
  \code{\link{print.solar_time}}, \code{\link{solar_time}},
  \code{\link{sun_angles}}
}
\concept{astronomy related functions}
