% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes_mcmc.R
\name{piv_MCMC}
\alias{piv_MCMC}
\title{JAGS Sampling for Gaussian Mixture Models and Clustering via Co-Association Matrix.}
\usage{
piv_MCMC(y, k, priors, nMC, piv.criterion = c("MUS", "maxsumint",
  "minsumnoint", "maxsumdiff"), clustering = c("diana", "hclust"))
}
\arguments{
\item{y}{N-dimensional data vector/matrix.}

\item{k}{Number of mixture components.}

\item{priors}{Input prior hyperparameters (see Details).}

\item{nMC}{Number of MCMC iterations for the JAGS function execution.}

\item{piv.criterion}{The pivotal criterion used for identifying one pivot
for each group. Possible choices are: \code{"MUS", "maxsumint", "minsumnoint",
"maxsumdiff"}.
If \code{k <= 4}, the default method is \code{"MUS"};
otherwise, the default method is \code{"maxsumdiff"} (see the Details and
the vignette).}

\item{clustering}{The clustering technique adopted for partitioning the
\code{N} observations into \code{k} groups. Possible choices: \code{"diana"} (default),
\code{"hclust"}.}
}
\value{
The function gives the MCMC output, the clustering
solutions and the pivotal indexes. Here there is a complete list of outputs.

\item{\code{Freq}}{  \code{k x 2} matrix where: the first column
reports the number of units allocated to each group
as given by JAGS program; the second
column reports the same number of units as given by the
chains' post-processing.}
\item{\code{true.iter}}{ The number of MCMC iterations for which
the number of JAGS groups exactly coincides with the prespecified
number of groups \code{k}.}
\item{\code{z} }{  \code{N x k x true.iter} array with values: 1,
if the \eqn{i}-th unit belongs to the \eqn{j}-th group at
the \eqn{h}-th iteration; 0, otherwise.}
\item{\code{ris}}{  MCMC output matrix as provided by JAGS.}
\item{\code{groupPost}}{ \code{true.iter x N} matrix
with values from \code{1:k} indicating the post-processed group allocation
vector.}
\item{\code{mu_switch}}{  If \code{y} is a vector, a \code{true.iter x k}
matrix with the post-processed MCMC chains for the mean parameters; if
\code{y} is a matrix, a \code{true.iter x 2 x k} array with
the post-processed MCMC chains for the mean parameters.}
\item{\code{mu_raw}}{ If \code{y} is a vector, a \code{nMC x k} matrix
with the raw MCMC chains for the mean parameters as given by JAGS; if
\code{y} is a matrix, a \code{nMC x 2 x k} array with the raw MCMC chains
for the mean parameters as given by JAGS.}
\item{\code{C}}{Co-association matrix constructed from the MCMC sample.}
\item{\code{grr}}{Group vector allocation as provided by
\code{"diana"} or \code{"hclust"}.}
\item{\code{pivots}}{ The pivotal units identified by the
selected pivotal criterion.}
\item{\code{piv.criterion}}{ Gives the pivotal criterion used for identifying
the pivots.}
}
\description{
Perform MCMC JAGS sampling for Gaussian mixture models, post-process the chains and apply a clustering technique to the MCMC sample. Pivotal units for each group are selected among four alternative criteria.
}
\details{
The function fits univariate and bivariate Bayesian Gaussian mixture models of the form
(here for univariate only):
\deqn{(Y_i|Z_i=j) \sim \mathcal{N}(\mu_j,\phi_j),}
where the \eqn{Z_i}, \eqn{i=1,\ldots,N}, are i.i.d. random variables, \eqn{j=1,\dots,k},
\eqn{\phi_j} is the group variance,  \eqn{Z_i \in {1,\ldots,k }} are the
latent group allocation, and
\deqn{P(Z_i=j)=\pi_j.}
The likelihood of the model is then
\deqn{L(y;\mu,\pi,\phi) = \prod_{i=1}^N \sum_{j=1}^k \pi_j \mathcal{N}(\mu_j,\phi_j),}
where \eqn{(\mu, \phi)=(\mu_{1},\dots,\mu_{k},\phi_{1},\ldots,\phi_{k})}
are the component-specific parameters and \eqn{\pi=(\pi_{1},\dots,\pi_{k})}
the mixture weights. Let \eqn{\nu} denote a permutation of \eqn{{ 1,\ldots,k }},
and let \eqn{\nu(\mu)= (\mu_{\nu(1)},\ldots,} \eqn{ \mu_{\nu(k)})},
\eqn{\nu(\phi)= (\phi_{\nu(1)},\ldots,} \eqn{ \phi_{\nu(k)})},
\eqn{ \nu(\pi)=(\pi_{\nu(1)},\ldots,\pi_{\nu(k)})} be the
corresponding permutations of \eqn{\mu}, \eqn{\phi} and \eqn{\pi}.
 Denote by \eqn{V} the set of all the permutations of the indexes
 \eqn{{1,\ldots,k }}, the likelihood above is invariant under any
 permutation \eqn{\nu \in V}, that is
\deqn{
L(y;\mu,\pi,\phi) = L(y;\nu(\mu),\nu(\pi),\nu(\phi)).}
As a consequence, the model is unidentified with respect to an
arbitrary permutation of the labels.
When Bayesian inference for the model is performed,
if the prior distribution \eqn{p_0(\mu,\pi,\phi)} is invariant under a permutation of the indices, then so is the posterior. That is, if \eqn{p_0(\mu,\pi,\phi) = p_0(\nu(\mu),\nu(\pi),\phi)}, then
\deqn{
p(\mu,\pi,\phi| y) \propto p_0(\mu,\pi,\phi)L(y;\mu,\pi,\phi)}
is multimodal with (at least) \eqn{k!} modes.

Priors are chosen as weakly informative. For univariate mixtures,
the specification is the same as the function \code{BMMmodel} of the
\code{bayesmix} package:

 \deqn{\mu_j \sim \mathcal{N}(0, 1/B0inv)}
 \deqn{\phi_j \sim \mbox{invGamma}(nu0Half, nu0S0Half)}
 \deqn{\pi \sim \mbox{Dirichlet}(1,\ldots,1)}
 \deqn{S0 \sim \mbox{Gamma}(g0Half, g0G0Half),}

 with default values: \eqn{B0inv=0.1, nu0Half =10, S0=2,
 nu0S0Half= nu0Half\times S0,
 g0Half = 5e-17, g0G0Half = 5e-33}, in accordance with the default
 specification:

 \code{priors=list(kind = "independence", parameter = "priorsFish",
 hierarchical = "tau")}

 (see \code{bayesmix} for further details and choices).

For bivariate mixtures, the prior specification is the following:

\deqn{ \bm{\mu}_j  \sim \mathcal{N}_2(\bm{\mu}_0, S2)}
\deqn{ 1/\Sigma \sim \mbox{Wishart}(S3, 3)}
\deqn{\pi \sim \mbox{Dirichlet}(1,\ldots,1),}

where \eqn{S2} and \eqn{S3} are diagonal matrices
with diagonal elements (the variances)
equal to 1e+05. The user may specify other values for the hyperparameters
\eqn{\bm{\mu}_0, S2, S3} via \code{priors} argument in such a way:

\code{priors =list(mu0 = c(1,1), S2 = ...,S3 = ...)},

with the constraint for \eqn{S2} and \eqn{S3} to be positive definite.

The function performs JAGS sampling using the \code{bayesmix} package for univariate Gaussian mixtures, and the \code{runjags}
package for bivariate Gaussian mixtures. After MCMC sampling, this function
clusters the units in \code{k} groups,
calls the \code{piv_sel()} function and yields the
pivots obtained from one among four different
methods (the user may specify one among them via \code{piv.criterion}
argument):
 \code{"maxsumint"}, \code{"minsumnoint"}, \code{"maxsumdiff"}
 and \code{"MUS"} (available only if \code{k <= 4})
(see the vignette for thorough details). Due to computational reasons
clarified in the Details section of the function \code{piv_rel}, the
length of the MCMC chains will be minor or equal than the input
argument \code{nMC}; this length, corresponding to the value
\code{true.iter} returned by the procedure, is the number of
MCMC iterations for which
the number of JAGS groups exactly coincides with the prespecified
number of groups \code{k}.
}
\examples{

# Bivariate simulation
\dontrun{
N   <- 200
k   <- 4
nMC <- 1000
M1  <-c(-.5,8)
M2  <- c(25.5,.1)
M3  <- c(49.5,8)
M4  <- c(63.0,.1)
Mu  <- matrix(rbind(M1,M2,M3,M4),c(4,2))
stdev    <- cbind(rep(1,k), rep(200,k))
Sigma.p1 <- matrix(c(stdev[1,1],0,0,stdev[1,1]), nrow=2, ncol=2)
Sigma.p2 <- matrix(c(stdev[1,2],0,0,stdev[1,2]), nrow=2, ncol=2)
W <- c(0.2,0.8)
sim <- piv_sim(N,k,Mu, stdev, Sigma.p1,Sigma.p2,W)
res <- piv_MCMC(y = sim$y, k =k, nMC = nMC)
#changing priors
res2 <- piv_MCMC(y = sim$y,
                 priors = list (
                 mu0=c(1,1),
                 S2 = matrix(c(0.002,0,0, 0.1),2,2, byrow=TRUE),
                 S3 = matrix(c(0.1,0,0,0.1), 2,2, byrow =TRUE)),
                 k = k, nMC = nMC)
}


# Fishery data (bayesmix package)
\dontrun{
data(fish)
y <- fish[,1]
k <- 5
nMC <- 5000
res <- piv_MCMC(y = y, k = k, nMC = nMC)
# changing priors
res2   <- piv_MCMC(y = y,
                   priors = list(kind = "condconjugate",
                   parameter = "priorsRaftery",
                   hierarchical = "tau"),  k =k, nMC = nMC)
}
}
\references{
Egidi, L., Pappadà, R., Pauli, F. and Torelli, N. (2018). Relabelling in Bayesian Mixture
Models by Pivotal Units. Statistics and Computing, 28(4), 957-969.
}
\author{
Leonardo Egidi \url{legidi@units.it}
}
