library(testthat)

# most common expectations:
# equality:        expect_equal() and expect_identical()
# regexp:          expect_match()
# catch-all:       expect_true() and expect_false()
# console output:  expect_output()
# messages:        expect_message()
# warning:         expect_warning()
# errors:          expect_error()

escapeString <- function(s) {
  t <- gsub("(\\\\)", "\\\\\\\\", s)
  t <- gsub("(\n)", "\\\\n", t)
  t <- gsub("(\r)", "\\\\r", t)
  t <- gsub("(\")", "\\\\\"", t)
  return(t)
}

prepStr <- function(s) {
  t <- escapeString(s)
  u <- eval(parse(text=paste0("\"", t, "\"")))
  if(s!=u) stop("Unable to escape string!")
  t <- paste0("\thtml <- \"", t, "\"")
  utils::writeClipboard(t)
  return(invisible())
}

evaluationMode <- "sequential"
processingLibrary <- "dplyr"
description <- "test: sequential dplyr"
countFunction <- "n()"
isDevelopmentVersion <- (length(strsplit(packageDescription("pivottabler")$Version, "\\.")[[1]]) > 3)

testScenarios <- function(description="test", releaseEvaluationMode="batch", releaseProcessingLibrary="dplyr", runAllForReleaseVersion=FALSE) {
  isDevelopmentVersion <- (length(strsplit(packageDescription("pivottabler")$Version, "\\.")[[1]]) > 3)
  if(isDevelopmentVersion||runAllForReleaseVersion) {
    evaluationModes <- c("sequential", "batch")
    processingLibraries <- c("dplyr", "data.table")
  }
  else {
    evaluationModes <- releaseEvaluationMode
    processingLibraries <- releaseProcessingLibrary
  }
  testCount <- length(evaluationModes)*length(processingLibraries)
  c1 <- character(testCount)
  c2 <- character(testCount)
  c3 <- character(testCount)
  c4 <- character(testCount)
  testCount <- 0
  for(evaluationMode in evaluationModes)
    for(processingLibrary in processingLibraries) {
      testCount <- testCount + 1
      c1[testCount] <- evaluationMode
      c2[testCount] <- processingLibrary
      c3[testCount] <- paste0(description, ": ", evaluationMode, " ", processingLibrary)
      c4[testCount] <- ifelse(processingLibrary=="data.table", ".N", "n()")
    }
  df <- data.frame(evaluationMode=c1, processingLibrary=c2, description=c3, countFunction=c4, stringsAsFactors=FALSE)
  return(df)
}


context("ALTERNATIVE LAYOUT TESTS")

scenarios <- testScenarios("Append groups at level 1 (top level)")
for(i in 1:nrow(scenarios)) {
  if(!isDevelopmentVersion) break
  evaluationMode <- scenarios$evaluationMode[i]
  processingLibrary <- scenarios$processingLibrary[i]
  description <- scenarios$description[i]
  countFunction <- scenarios$countFunction[i]

  test_that(description, {

    library(pivottabler)
    pt <- PivotTable$new(processingLibrary=processingLibrary, evaluationMode=evaluationMode, compatibility=list(noDataGroupNBSP=TRUE))
    pt$addData(bhmtrains)
    pt$addColumnDataGroups("TrainCategory")
    pt$addRowDataGroups("TOC")
    pt$addRowDataGroups("PowerType")
    pt$addRowDataGroups("SchedSpeedMPH")
    grpLevelCheck <- pt$addRowDataGroups("Status", atLevel=1)[[1]]$levelNumber
    pt$defineCalculation(calculationName="TotalTrains", summariseExpression=countFunction)
    pt$evaluatePivot()
    # pt$renderPivot()
    # sum(pt$cells$asMatrix(), na.rm=TRUE)
    # prepStr(as.character(pt$getHtml()))
    html <- "<table class=\"Table\">\n  <tr>\n    <th class=\"RowHeader\" colspan=\"3\">&nbsp;</th>\n    <th class=\"ColumnHeader\">Express Passenger</th>\n    <th class=\"ColumnHeader\">Ordinary Passenger</th>\n    <th class=\"ColumnHeader\">Total</th>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"5\">Arriva Trains Wales</th>\n    <th class=\"RowHeader\" rowspan=\"4\">DMU</th>\n    <th class=\"RowHeader\">75</th>\n    <td class=\"Cell\">63</td>\n    <td class=\"Cell\">58</td>\n    <td class=\"Total\">121</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\">2989</td>\n    <td class=\"Cell\">772</td>\n    <td class=\"Total\">3761</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">27</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">27</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"8\">CrossCountry</th>\n    <th class=\"RowHeader\" rowspan=\"4\">DMU</th>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">10961</td>\n    <td class=\"Cell\">63</td>\n    <td class=\"Total\">11024</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">11040</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">11040</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">132</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">132</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">22133</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22196</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">HST</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">724</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">724</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">8</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">8</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">732</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">732</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22865</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22928</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"11\">London Midland</th>\n    <th class=\"RowHeader\" rowspan=\"5\">DMU</th>\n    <th class=\"RowHeader\">75</th>\n    <td class=\"Cell\">845</td>\n    <td class=\"Cell\">5259</td>\n    <td class=\"Total\">6104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\">2340</td>\n    <td class=\"Cell\">59</td>\n    <td class=\"Total\">2399</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">2450</td>\n    <td class=\"Cell\">273</td>\n    <td class=\"Total\">2723</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">3</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">5638</td>\n    <td class=\"Total\">5591</td>\n    <td class=\"Total\">11229</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"5\">EMU</th>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">17801</td>\n    <td class=\"Total\">17801</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">7820</td>\n    <td class=\"Cell\">10384</td>\n    <td class=\"Total\">18204</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">110</th>\n    <td class=\"Cell\">1020</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1020</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">9</td>\n    <td class=\"Cell\">16</td>\n    <td class=\"Total\">25</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">8849</td>\n    <td class=\"Total\">28201</td>\n    <td class=\"Total\">37050</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">14487</td>\n    <td class=\"Total\">33792</td>\n    <td class=\"Total\">48279</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"7\">Virgin Trains</th>\n    <th class=\"RowHeader\" rowspan=\"3\">DMU</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">2135</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2135</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2137</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2137</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">EMU</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">6452</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6452</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">5</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">6457</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6457</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8594</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8594</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">49025</td>\n    <td class=\"Total\">34685</td>\n    <td class=\"Total\">83710</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">A</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">47780</td>\n    <td class=\"Cell\">33726</td>\n    <td class=\"Total\">81506</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">1190</td>\n    <td class=\"Cell\">931</td>\n    <td class=\"Total\">2121</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">55</td>\n    <td class=\"Cell\">28</td>\n    <td class=\"Total\">83</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">49025</td>\n    <td class=\"Total\">34685</td>\n    <td class=\"Total\">83710</td>\n  </tr>\n</table>"

    expect_equal(sum(pt$cells$asMatrix(), na.rm=TRUE), 1004520)
    expect_equal(grpLevelCheck, 1)
    expect_identical(as.character(pt$getHtml()), html)
  })
}



scenarios <- testScenarios("Append groups at level 2 (intermediate level)")
for(i in 1:nrow(scenarios)) {
  evaluationMode <- scenarios$evaluationMode[i]
  processingLibrary <- scenarios$processingLibrary[i]
  description <- scenarios$description[i]
  countFunction <- scenarios$countFunction[i]

  test_that(description, {

    skip_on_cran()

    library(pivottabler)
    pt <- PivotTable$new(processingLibrary=processingLibrary, evaluationMode=evaluationMode, compatibility=list(noDataGroupNBSP=TRUE))
    pt$addData(bhmtrains)
    pt$addColumnDataGroups("TrainCategory")
    pt$addRowDataGroups("TOC")
    pt$addRowDataGroups("PowerType")
    pt$addRowDataGroups("SchedSpeedMPH")
    grpLevelCheck <- pt$addRowDataGroups("Status", atLevel=2)[[1]]$levelNumber
    pt$defineCalculation(calculationName="TotalTrains", summariseExpression=countFunction)
    pt$evaluatePivot()
    # pt$renderPivot()
    # sum(pt$cells$asMatrix(), na.rm=TRUE)
    # prepStr(as.character(pt$getHtml()))
    html <- "<table class=\"Table\">\n  <tr>\n    <th class=\"RowHeader\" colspan=\"3\">&nbsp;</th>\n    <th class=\"ColumnHeader\">Express Passenger</th>\n    <th class=\"ColumnHeader\">Ordinary Passenger</th>\n    <th class=\"ColumnHeader\">Total</th>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"9\">Arriva Trains Wales</th>\n    <th class=\"RowHeader\" rowspan=\"4\">DMU</th>\n    <th class=\"RowHeader\">75</th>\n    <td class=\"Cell\">63</td>\n    <td class=\"Cell\">58</td>\n    <td class=\"Total\">121</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\">2989</td>\n    <td class=\"Cell\">772</td>\n    <td class=\"Total\">3761</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">27</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">27</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">A</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">3018</td>\n    <td class=\"Cell\">815</td>\n    <td class=\"Total\">3833</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">59</td>\n    <td class=\"Cell\">15</td>\n    <td class=\"Total\">74</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"12\">CrossCountry</th>\n    <th class=\"RowHeader\" rowspan=\"4\">DMU</th>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">10961</td>\n    <td class=\"Cell\">63</td>\n    <td class=\"Total\">11024</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">11040</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">11040</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">132</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">132</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">22133</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22196</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">HST</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">724</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">724</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">8</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">8</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">732</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">732</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22865</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22928</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">A</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">22270</td>\n    <td class=\"Cell\">60</td>\n    <td class=\"Total\">22330</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">569</td>\n    <td class=\"Cell\">2</td>\n    <td class=\"Total\">571</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">26</td>\n    <td class=\"Cell\">1</td>\n    <td class=\"Total\">27</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22865</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22928</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"15\">London Midland</th>\n    <th class=\"RowHeader\" rowspan=\"5\">DMU</th>\n    <th class=\"RowHeader\">75</th>\n    <td class=\"Cell\">845</td>\n    <td class=\"Cell\">5259</td>\n    <td class=\"Total\">6104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\">2340</td>\n    <td class=\"Cell\">59</td>\n    <td class=\"Total\">2399</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">2450</td>\n    <td class=\"Cell\">273</td>\n    <td class=\"Total\">2723</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">3</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">5638</td>\n    <td class=\"Total\">5591</td>\n    <td class=\"Total\">11229</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"5\">EMU</th>\n    <th class=\"RowHeader\">90</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">17801</td>\n    <td class=\"Total\">17801</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">100</th>\n    <td class=\"Cell\">7820</td>\n    <td class=\"Cell\">10384</td>\n    <td class=\"Total\">18204</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">110</th>\n    <td class=\"Cell\">1020</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1020</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">9</td>\n    <td class=\"Cell\">16</td>\n    <td class=\"Total\">25</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">8849</td>\n    <td class=\"Total\">28201</td>\n    <td class=\"Total\">37050</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">14487</td>\n    <td class=\"Total\">33792</td>\n    <td class=\"Total\">48279</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">A</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">14133</td>\n    <td class=\"Cell\">32851</td>\n    <td class=\"Total\">46984</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">336</td>\n    <td class=\"Cell\">914</td>\n    <td class=\"Total\">1250</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">18</td>\n    <td class=\"Cell\">27</td>\n    <td class=\"Total\">45</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">14487</td>\n    <td class=\"Total\">33792</td>\n    <td class=\"Total\">48279</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"11\">Virgin Trains</th>\n    <th class=\"RowHeader\" rowspan=\"3\">DMU</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">2135</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2135</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2137</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2137</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">EMU</th>\n    <th class=\"RowHeader\">125</th>\n    <td class=\"Cell\">6452</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6452</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">NA</th>\n    <td class=\"Cell\">5</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">6457</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6457</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8594</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8594</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">A</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">8359</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">8359</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">226</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">226</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Cell\">9</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">9</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8594</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8594</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">49025</td>\n    <td class=\"Total\">34685</td>\n    <td class=\"Total\">83710</td>\n  </tr>\n</table>"

    expect_equal(sum(pt$cells$asMatrix(), na.rm=TRUE), 1004520)
    expect_equal(grpLevelCheck, 2)
    expect_identical(as.character(pt$getHtml()), html)
  })
}



scenarios <- testScenarios("Append groups at level 4 (leaf level)")
for(i in 1:nrow(scenarios)) {
  if(!isDevelopmentVersion) break
  evaluationMode <- scenarios$evaluationMode[i]
  processingLibrary <- scenarios$processingLibrary[i]
  description <- scenarios$description[i]
  countFunction <- scenarios$countFunction[i]

  test_that(description, {

    library(pivottabler)
    pt <- PivotTable$new(processingLibrary=processingLibrary, evaluationMode=evaluationMode, compatibility=list(noDataGroupNBSP=TRUE))
    pt$addData(bhmtrains)
    pt$addColumnDataGroups("TrainCategory")
    pt$addRowDataGroups("TOC")
    pt$addRowDataGroups("PowerType")
    pt$addRowDataGroups("SchedSpeedMPH")
    grpLevelCheck <- pt$addRowDataGroups("Status", atLevel=4)[[1]]$levelNumber
    pt$defineCalculation(calculationName="TotalTrains", summariseExpression=countFunction)
    pt$evaluatePivot()
    # pt$renderPivot()
    # sum(pt$cells$asMatrix(), na.rm=TRUE)
    # prepStr(as.character(pt$getHtml()))
    html <- "<table class=\"Table\">\n  <tr>\n    <th class=\"RowHeader\" colspan=\"4\">&nbsp;</th>\n    <th class=\"ColumnHeader\">Express Passenger</th>\n    <th class=\"ColumnHeader\">Ordinary Passenger</th>\n    <th class=\"ColumnHeader\">Total</th>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"12\">Arriva Trains Wales</th>\n    <th class=\"RowHeader\" rowspan=\"11\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"3\">75</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">62</td>\n    <td class=\"Cell\">58</td>\n    <td class=\"Total\">120</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">58</td>\n    <td class=\"Total\">121</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2933</td>\n    <td class=\"Cell\">757</td>\n    <td class=\"Total\">3690</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">54</td>\n    <td class=\"Cell\">15</td>\n    <td class=\"Total\">69</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2989</td>\n    <td class=\"Total\">772</td>\n    <td class=\"Total\">3761</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">23</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">23</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">4</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">4</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">27</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">27</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"20\">CrossCountry</th>\n    <th class=\"RowHeader\" rowspan=\"12\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">10743</td>\n    <td class=\"Cell\">60</td>\n    <td class=\"Total\">10803</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">208</td>\n    <td class=\"Cell\">2</td>\n    <td class=\"Total\">210</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">10</td>\n    <td class=\"Cell\">1</td>\n    <td class=\"Total\">11</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">10961</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">11024</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">10714</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">10714</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">310</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">310</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">16</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">16</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">11040</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">11040</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">104</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">28</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">28</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">132</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">132</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22133</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22196</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"7\">HST</th>\n    <th class=\"RowHeader\" rowspan=\"3\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">703</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">703</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">21</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">21</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">724</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">724</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">6</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">8</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">732</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">732</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22865</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22928</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"31\">London Midland</th>\n    <th class=\"RowHeader\" rowspan=\"14\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">75</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">822</td>\n    <td class=\"Cell\">5192</td>\n    <td class=\"Total\">6014</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">22</td>\n    <td class=\"Cell\">63</td>\n    <td class=\"Total\">85</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\">4</td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">845</td>\n    <td class=\"Total\">5259</td>\n    <td class=\"Total\">6104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2305</td>\n    <td class=\"Cell\">59</td>\n    <td class=\"Total\">2364</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">33</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">33</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2340</td>\n    <td class=\"Total\">59</td>\n    <td class=\"Total\">2399</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2404</td>\n    <td class=\"Cell\">269</td>\n    <td class=\"Total\">2673</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">46</td>\n    <td class=\"Cell\">4</td>\n    <td class=\"Total\">50</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2450</td>\n    <td class=\"Total\">273</td>\n    <td class=\"Total\">2723</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">3</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">3</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">5638</td>\n    <td class=\"Total\">5591</td>\n    <td class=\"Total\">11229</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"16\">EMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">17130</td>\n    <td class=\"Total\">17130</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">660</td>\n    <td class=\"Total\">660</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">11</td>\n    <td class=\"Total\">11</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">17801</td>\n    <td class=\"Total\">17801</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">7609</td>\n    <td class=\"Cell\">10186</td>\n    <td class=\"Total\">17795</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">197</td>\n    <td class=\"Cell\">186</td>\n    <td class=\"Total\">383</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">14</td>\n    <td class=\"Cell\">12</td>\n    <td class=\"Total\">26</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">7820</td>\n    <td class=\"Total\">10384</td>\n    <td class=\"Total\">18204</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">110</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">983</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">983</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">36</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">36</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">1020</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">1020</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">7</td>\n    <td class=\"Cell\">15</td>\n    <td class=\"Total\">22</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\">1</td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">9</td>\n    <td class=\"Total\">16</td>\n    <td class=\"Total\">25</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8849</td>\n    <td class=\"Total\">28201</td>\n    <td class=\"Total\">37050</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">14487</td>\n    <td class=\"Total\">33792</td>\n    <td class=\"Total\">48279</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"15\">Virgin Trains</th>\n    <th class=\"RowHeader\" rowspan=\"7\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2026</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2026</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">107</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">107</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2135</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2135</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">2137</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2137</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"7\">EMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">6326</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6326</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">119</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">119</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">7</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">7</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">6452</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6452</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">5</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">5</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">6457</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6457</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8594</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8594</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">49025</td>\n    <td class=\"Total\">34685</td>\n    <td class=\"Total\">83710</td>\n  </tr>\n</table>"

    expect_equal(sum(pt$cells$asMatrix(), na.rm=TRUE), 837100)
    expect_equal(grpLevelCheck, 4)
    expect_identical(as.character(pt$getHtml()), html)
  })
}



scenarios <- testScenarios("Append groups at level 5 (beyond leaf level)")  # check that a level number that is too high gets clamped back to the leaf level
for(i in 1:nrow(scenarios)) {
  if(!isDevelopmentVersion) break
  evaluationMode <- scenarios$evaluationMode[i]
  processingLibrary <- scenarios$processingLibrary[i]
  description <- scenarios$description[i]
  countFunction <- scenarios$countFunction[i]

  test_that(description, {

    library(pivottabler)
    pt <- PivotTable$new(processingLibrary=processingLibrary, evaluationMode=evaluationMode, compatibility=list(noDataGroupNBSP=TRUE))
    pt$addData(bhmtrains)
    pt$addColumnDataGroups("TrainCategory")
    pt$addRowDataGroups("TOC")
    pt$addRowDataGroups("PowerType")
    pt$addRowDataGroups("SchedSpeedMPH")
    grpLevelCheck <- pt$addRowDataGroups("Status", atLevel=5)[[1]]$levelNumber
    pt$defineCalculation(calculationName="TotalTrains", summariseExpression=countFunction)
    pt$evaluatePivot()
    # pt$renderPivot()
    # sum(pt$cells$asMatrix(), na.rm=TRUE)
    # prepStr(as.character(pt$getHtml()))
    html <- "<table class=\"Table\">\n  <tr>\n    <th class=\"RowHeader\" colspan=\"4\">&nbsp;</th>\n    <th class=\"ColumnHeader\">Express Passenger</th>\n    <th class=\"ColumnHeader\">Ordinary Passenger</th>\n    <th class=\"ColumnHeader\">Total</th>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"12\">Arriva Trains Wales</th>\n    <th class=\"RowHeader\" rowspan=\"11\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"3\">75</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">62</td>\n    <td class=\"Cell\">58</td>\n    <td class=\"Total\">120</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">58</td>\n    <td class=\"Total\">121</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2933</td>\n    <td class=\"Cell\">757</td>\n    <td class=\"Total\">3690</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">54</td>\n    <td class=\"Cell\">15</td>\n    <td class=\"Total\">69</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2989</td>\n    <td class=\"Total\">772</td>\n    <td class=\"Total\">3761</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">23</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">23</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">4</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">4</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">27</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">27</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">3079</td>\n    <td class=\"Total\">830</td>\n    <td class=\"Total\">3909</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"20\">CrossCountry</th>\n    <th class=\"RowHeader\" rowspan=\"12\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">10743</td>\n    <td class=\"Cell\">60</td>\n    <td class=\"Total\">10803</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">208</td>\n    <td class=\"Cell\">2</td>\n    <td class=\"Total\">210</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">10</td>\n    <td class=\"Cell\">1</td>\n    <td class=\"Total\">11</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">10961</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">11024</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">10714</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">10714</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">310</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">310</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">16</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">16</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">11040</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">11040</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">104</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">28</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">28</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">132</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">132</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22133</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22196</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"7\">HST</th>\n    <th class=\"RowHeader\" rowspan=\"3\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">703</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">703</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">21</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">21</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">724</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">724</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">6</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">8</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">732</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">732</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">22865</td>\n    <td class=\"Total\">63</td>\n    <td class=\"Total\">22928</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"31\">London Midland</th>\n    <th class=\"RowHeader\" rowspan=\"14\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">75</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">822</td>\n    <td class=\"Cell\">5192</td>\n    <td class=\"Total\">6014</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">22</td>\n    <td class=\"Cell\">63</td>\n    <td class=\"Total\">85</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\">4</td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">845</td>\n    <td class=\"Total\">5259</td>\n    <td class=\"Total\">6104</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2305</td>\n    <td class=\"Cell\">59</td>\n    <td class=\"Total\">2364</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">33</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">33</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2340</td>\n    <td class=\"Total\">59</td>\n    <td class=\"Total\">2399</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2404</td>\n    <td class=\"Cell\">269</td>\n    <td class=\"Total\">2673</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">46</td>\n    <td class=\"Cell\">4</td>\n    <td class=\"Total\">50</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2450</td>\n    <td class=\"Total\">273</td>\n    <td class=\"Total\">2723</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">3</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">3</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">5638</td>\n    <td class=\"Total\">5591</td>\n    <td class=\"Total\">11229</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"16\">EMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">90</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">17130</td>\n    <td class=\"Total\">17130</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">660</td>\n    <td class=\"Total\">660</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\"></td>\n    <td class=\"Cell\">11</td>\n    <td class=\"Total\">11</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">17801</td>\n    <td class=\"Total\">17801</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">100</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">7609</td>\n    <td class=\"Cell\">10186</td>\n    <td class=\"Total\">17795</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">197</td>\n    <td class=\"Cell\">186</td>\n    <td class=\"Total\">383</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">14</td>\n    <td class=\"Cell\">12</td>\n    <td class=\"Total\">26</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">7820</td>\n    <td class=\"Total\">10384</td>\n    <td class=\"Total\">18204</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"4\">110</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">983</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">983</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">36</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">36</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">1</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">1</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">1020</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">1020</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"3\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">7</td>\n    <td class=\"Cell\">15</td>\n    <td class=\"Total\">22</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\">1</td>\n    <td class=\"Total\">3</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">9</td>\n    <td class=\"Total\">16</td>\n    <td class=\"Total\">25</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8849</td>\n    <td class=\"Total\">28201</td>\n    <td class=\"Total\">37050</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">14487</td>\n    <td class=\"Total\">33792</td>\n    <td class=\"Total\">48279</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"15\">Virgin Trains</th>\n    <th class=\"RowHeader\" rowspan=\"7\">DMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2026</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2026</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">107</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">107</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2135</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2135</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">2</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">2</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">2137</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">2137</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"7\">EMU</th>\n    <th class=\"RowHeader\" rowspan=\"4\">125</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">6326</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">6326</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">C</th>\n    <td class=\"Cell\">119</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">119</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">R</th>\n    <td class=\"Cell\">7</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">7</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">6452</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6452</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\" rowspan=\"2\">NA</th>\n    <th class=\"RowHeader\">A</th>\n    <td class=\"Cell\">5</td>\n    <td class=\"Cell\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <td class=\"Total\">5</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">5</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">6457</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">6457</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">8594</td>\n    <td class=\"Total\"></td>\n    <td class=\"Total\">8594</td>\n  </tr>\n  <tr>\n    <th class=\"RowHeader\">Total</th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <th class=\"RowHeader\"></th>\n    <td class=\"Total\">49025</td>\n    <td class=\"Total\">34685</td>\n    <td class=\"Total\">83710</td>\n  </tr>\n</table>"

    expect_equal(sum(pt$cells$asMatrix(), na.rm=TRUE), 837100)
    expect_equal(grpLevelCheck, 4)
    expect_identical(as.character(pt$getHtml()), html)
  })
}



