% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_yaml.R
\name{agent_yaml_interrogate}
\alias{agent_yaml_interrogate}
\title{Read a YAML file to interrogate a target table immediately}
\usage{
agent_yaml_interrogate(path)
}
\arguments{
\item{path}{A path to a YAML file that specifies a validation plan for an
\emph{agent}.}
}
\description{
The \code{agent_yaml_interrogate()} function operates much like the
\code{\link[=agent_yaml_read]{agent_yaml_read()}} function (reading a \strong{pointblank} YAML file and
generating an \emph{agent} with a validation plan in place). The key difference is
that this function takes things a step function and interrogates the target
table (defined by table-reading, \code{read_fn}, function that is required in the
YAML file). The additional auto-invocation of \code{\link[=interrogate]{interrogate()}} uses the
default options of that function. As with \code{\link[=agent_yaml_read]{agent_yaml_read()}} the agent is
returned except, this time, it has intel from the interrogation.
}
\section{Function ID}{

7-3
}

\examples{
# Let's go through the process of
# developing an agent with a validation
# plan (to be used for the data quality
# analysis of the `small_table` dataset),
# and then offloading that validation
# plan to a pointblank YAML file; this
# will later be read in as a new agent and
# the target data will be interrogated
# (one step) with `agent_yaml_interrogate()`

# We ought to think about what's
# tolerable in terms of data quality so
# let's designate proportional failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )

# Now create a pointblank `agent` object
# and give it the `al` object (which
# serves as a default for all validation
# steps which can be overridden); the
# data will be referenced in a `read_fn`
# (a requirement for writing to YAML)
agent <- 
  create_agent(
    read_fn = ~small_table,
    name = "example",
    actions = al
  )

# Then, as with any `agent` object, we
# can add steps to the validation plan by
# using as many validation functions as we
# want
agent <-
  agent \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(vars(d), 100) \%>\%
  col_vals_lte(vars(c), 5)

# The agent can be written to a pointblank
# YAML file with `agent_yaml_write()`
# agent_yaml_write(agent, filename = "x.yml")

# The 'x.yml' file is available in the package
# through `system.file()`
yml_file <- 
  system.file("x.yml", package = "pointblank")

# We can view the YAML file in the console
# with the `agent_yaml_string()` function
agent_yaml_string(path = yml_file)

# The YAML can also be printed in the console
# by supplying the agent as the input
agent_yaml_string(agent = agent)

# We can interrogate the data (which
# is accessible through the `read_fn`)
# through direct use of the YAML file
# with `agent_yaml_interrogate()`
agent <- 
  agent_yaml_interrogate(path = yml_file)

class(agent)

# If it's desired to only create a new
# agent with the validation plan in place
# (stopping short of interrogating the data),
# then the `agent_yaml_read()` function
# will be useful
agent <- agent_yaml_read(path = yml_file)

class(agent)

}
\seealso{
Other pointblank YAML: 
\code{\link{agent_yaml_read}()},
\code{\link{agent_yaml_show_exprs}()},
\code{\link{agent_yaml_string}()},
\code{\link{agent_yaml_write}()}
}
\concept{pointblank YAML}
