% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_yaml.R
\name{agent_yaml_read}
\alias{agent_yaml_read}
\title{Read a YAML file to create a new agent with a validation plan}
\usage{
agent_yaml_read(path)
}
\arguments{
\item{path}{A path to a YAML file that specifies a validation plan for an
\emph{agent}.}
}
\description{
With \code{agent_yaml_read()} we can read a \strong{pointblank} YAML file that
describes a validation plan to be carried out by an \emph{agent} (typically
generated by the \code{\link[=agent_yaml_write]{agent_yaml_write()}} function. What's returned is a new
\emph{agent} with that validation plan, ready to interrogate the target table at
will (using the table-reading function stored as the \code{read_fn}). The agent
can be given more validation steps if needed before using \code{\link[=interrogate]{interrogate()}} or
taking part in any other agent ops (e.g., writing to disk with outputs intact
via \code{\link[=agent_write]{agent_write()}} or again to \strong{pointblank} YAML with
\code{\link[=agent_yaml_write]{agent_yaml_write()}}).

To get a picture of how \code{agent_yaml_read()} is interpreting the validation
plan specified in the \strong{pointblank} YAML, we can use the
\code{\link[=agent_yaml_show_exprs]{agent_yaml_show_exprs()}} function. That function shows us (in the console)
the \strong{pointblank} expressions for generating the described validation plan.
}
\section{Function ID}{

7-2
}

\examples{
# Let's go through the process of
# developing an agent with a validation
# plan (to be used for the data quality
# analysis of the `small_table` dataset),
# and then offloading that validation
# plan to a pointblank YAML file; this
# will be read in with `agent_yaml_read()`

# We ought to think about what's
# tolerable in terms of data quality so
# let's designate proportional failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )

# Now create a pointblank `agent` object
# and give it the `al` object (which
# serves as a default for all validation
# steps which can be overridden); the
# data will be referenced in a `read_fn`
# (a requirement for writing to YAML)
agent <- 
  create_agent(
    read_fn = ~small_table,
    name = "example",
    actions = al
  )

# Then, as with any `agent` object, we
# can add steps to the validation plan by
# using as many validation functions as we
# want
agent <-
  agent \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(vars(d), 100) \%>\%
  col_vals_lte(vars(c), 5)

# The agent can be written to a pointblank
# YAML file with `agent_yaml_write()`
# agent_yaml_write(agent, filename = "x.yml")

# The 'x.yml' file is available in the package
# through `system.file()`
yml_file <- 
  system.file("x.yml", package = "pointblank")

# We can view the YAML file in the console
# with the `agent_yaml_string()` function
agent_yaml_string(path = yml_file)

# The YAML can also be printed in the console
# by supplying the agent as the input
agent_yaml_string(agent = agent)

# At a later time, the YAML file can
# be read into a new agent with the
# `agent_yaml_read()` function
agent <- agent_yaml_read(path = yml_file)

class(agent)

# We can interrogate the data (which
# is accessible through the `read_fn`)
# with `interrogate()` and get an
# agent with intel, or, we can
# interrogate directly from the YAML
# file with `agent_yaml_interrogate()`
agent <- 
  agent_yaml_interrogate(path = yml_file)

class(agent)

}
\seealso{
Other pointblank YAML: 
\code{\link{agent_yaml_interrogate}()},
\code{\link{agent_yaml_show_exprs}()},
\code{\link{agent_yaml_string}()},
\code{\link{agent_yaml_write}()}
}
\concept{pointblank YAML}
