% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hybrid_policy_tree.R
\name{hybrid_policy_tree}
\alias{hybrid_policy_tree}
\title{Hybrid tree search}
\usage{
hybrid_policy_tree(
  X,
  Gamma,
  depth = 3,
  search.depth = 2,
  split.step = 1,
  min.node.size = 1,
  verbose = TRUE
)
}
\arguments{
\item{X}{The covariates used. Dimension \eqn{N*p} where \eqn{p} is the number of features.}

\item{Gamma}{The rewards for each action. Dimension \eqn{N*d} where \eqn{d} is the number of actions.}

\item{depth}{The depth of the fitted tree. Default is 3.}

\item{search.depth}{Depth to look ahead when splitting. Default is 2.}

\item{split.step}{An optional approximation parameter, the number of possible splits
to consider when performing tree search. split.step = 1 (default) considers every possible split, split.step = 10
considers splitting at every 10'th sample and may yield a substantial speedup for dense features.
Manually rounding or re-encoding continuous covariates with very high cardinality in a
problem specific manner allows for finer-grained control of the accuracy/runtime tradeoff and may in some cases
be the preferred approach.}

\item{min.node.size}{An integer indicating the smallest terminal node size permitted. Default is 1.}

\item{verbose}{Give verbose output. Default is TRUE.}
}
\value{
A policy_tree object.
}
\description{
Finds a depth k tree by looking ahead l steps.
}
\details{
Builds deeper trees by iteratively using exact tree search to look ahead l splits. For example,
with \code{depth = 3} and \code{search.depth = 2}, the root split is determined by a depth 2 exact tree,
and two new depth 2 trees are fit in the two immediate children using exact tree search,
leading to a total depth of 3 (the resulting tree may be shallower than the
specified \code{depth} depending on whether leaf nodes were pruned or not).
This algorithm scales with some coefficient multiple of the runtime of a \code{search.depth} \code{policy_tree},
which means that for this approach to be feasible it needs an (n, p, d) configuration in which
a \code{search.depth} \code{policy_tree} runs in reasonable time.

The algorithm: desired depth is given by \code{depth}. Each node is split using exact tree search
with depth  = \code{search.depth}. When we reach a node where the current level + \code{search.depth} is equal to \code{depth},
we stop and attach the \code{search.depth} subtree to this node.
We also stop if the best \code{search.depth} split yielded a leaf node.
}
\examples{
\donttest{
# Fit a depth three tree on doubly robust treatment effect estimates from a causal forest.
n <- 1500
p <- 5
X <- round(matrix(rnorm(n * p), n, p), 2)
W <- rbinom(n, 1, 1 / (1 + exp(X[, 3])))
tau <- 1 / (1 + exp((X[, 1] + X[, 2]) / 2)) - 0.5
Y <- X[, 3] + W * tau + rnorm(n)
c.forest <- grf::causal_forest(X, Y, W)
dr.scores <- double_robust_scores(c.forest)

tree <- hybrid_policy_tree(X, dr.scores, depth = 3)

# Predict treatment assignment.
predicted <- predict(tree, X)
}
}
