% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_SARSOP.R
\name{solve_SARSOP}
\alias{solve_SARSOP}
\title{Solve a POMDP Problem using SARSOP}
\usage{
solve_SARSOP(
  model,
  horizon = Inf,
  discount = NULL,
  terminal_values = NULL,
  method = "sarsop",
  digits = 7,
  parameter = NULL,
  verbose = FALSE
)
}
\arguments{
\item{model}{a POMDP problem specification created with \code{\link[=POMDP]{POMDP()}}.
Alternatively, a POMDP file or the URL for a POMDP file can be specified.}

\item{horizon}{need to be \code{Inf}.}

\item{discount}{discount factor in range \eqn{[0, 1]}. If \code{NULL}, then the
discount factor specified in \code{model} will be used.}

\item{terminal_values}{needs to be \code{NULL}. SARSOP does not use terminal values.}

\item{method}{string; there is only one method available called \code{"sarsop"}.}

\item{digits}{precision used when writing POMDP files (see
\code{\link[=write_POMDP]{write_POMDP()}}).}

\item{parameter}{a list with parameters passed on to
the function \code{sarsop} in package \pkg{sarsop}.}

\item{verbose}{logical, if set to \code{TRUE}, the function provides the
output of the solver in the R console.}
}
\value{
The solver returns an object of class POMDP which is a list with the
model specifications (\code{'model'}), the solution (\code{'solution'}), and the
solver output (\code{'solver_output'}).
}
\description{
This function uses the C++ implementation of the SARSOP algorithm
by Kurniawati, Hsu and Lee (2008) interfaced in
package \pkg{sarsop}
to solve infinite horizon problems that are formulated as partially observable Markov
decision processes (POMDPs). The result is an optimal or approximately
optimal policy.
}
\examples{
\dontrun{
# Solving the simple infinite-horizon Tiger problem with SARSOP
# You need to install package "sarsop"
data("Tiger")
Tiger

sol <- solve_SARSOP(model = Tiger)
sol

# look at solver output
sol$solver_output

# policy (value function (alpha vectors), optimal action and observation dependent transitions)
policy(sol)

# value function
plot_value_function(sol, ylim = c(0,20))

# plot the policy graph
plot_policy_graph(sol)

# reward of the optimal policy
reward(sol)

# Solve a problem specified as a POMDP file
sol <- solve_SARSOP("http://www.pomdp.org/examples/cheese.95.POMDP")
sol
}

}
\references{
Carl Boettiger, Jeroen Ooms and Milad Memarzadeh (2020). sarsop:
Approximate POMDP Planning Software. R package version 0.6.6.
https://CRAN.R-project.org/package=sarsop

H. Kurniawati, D. Hsu, and W.S. Lee (2008). SARSOP: Efficient point-based POMDP planning by approximating optimally reachable belief spaces. In Proc. Robotics: Science and Systems.
}
\seealso{
Other policy: 
\code{\link{optimal_action}()},
\code{\link{plot_policy_graph}()},
\code{\link{plot_value_function}()},
\code{\link{policy}()},
\code{\link{reward}()},
\code{\link{solve_POMDP}()}

Other solver: 
\code{\link{solve_MDP}()},
\code{\link{solve_POMDP}()}

Other POMDP: 
\code{\link{POMDP}()},
\code{\link{plot_belief_space}()},
\code{\link{sample_belief_space}()},
\code{\link{simulate_POMDP}()},
\code{\link{solve_POMDP}()},
\code{\link{transition_matrix}()},
\code{\link{update_belief}()},
\code{\link{write_POMDP}()}
}
\author{
Michael Hahsler
}
\concept{POMDP}
\concept{policy}
\concept{solver}
