% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/problem.R
\name{problem}
\alias{problem}
\alias{problem,Raster,Raster-method}
\alias{problem,Spatial,Raster-method}
\alias{problem,data.frame,data.frame-method}
\alias{problem,numeric,data.frame-method}
\alias{problem,data.frame,character-method}
\alias{problem,Spatial,character-method}
\alias{problem,Raster,ZonesRaster-method}
\alias{problem,Spatial,ZonesRaster-method}
\alias{problem,Spatial,ZonesCharacter-method}
\alias{problem,data.frame,ZonesCharacter-method}
\alias{problem,matrix,data.frame-method}
\alias{problem,sf,Raster-method}
\alias{problem,sf,ZonesCharacter-method}
\alias{problem,sf,character-method}
\alias{problem,sf,ZonesRaster-method}
\title{Conservation planning problem}
\usage{
problem(x, features, ...)

\S4method{problem}{Raster,Raster}(x, features, run_checks, ...)

\S4method{problem}{Raster,ZonesRaster}(x, features, run_checks, ...)

\S4method{problem}{Spatial,Raster}(x, features, cost_column, run_checks, ...)

\S4method{problem}{Spatial,ZonesRaster}(x, features, cost_column, run_checks, ...)

\S4method{problem}{Spatial,character}(x, features, cost_column, ...)

\S4method{problem}{Spatial,ZonesCharacter}(x, features, cost_column, ...)

\S4method{problem}{data.frame,character}(x, features, cost_column, ...)

\S4method{problem}{data.frame,ZonesCharacter}(x, features, cost_column, ...)

\S4method{problem}{data.frame,data.frame}(x, features, rij, cost_column, zones, ...)

\S4method{problem}{numeric,data.frame}(x, features, rij_matrix, ...)

\S4method{problem}{matrix,data.frame}(x, features, rij_matrix, ...)

\S4method{problem}{sf,Raster}(x, features, cost_column, run_checks, ...)

\S4method{problem}{sf,ZonesRaster}(x, features, cost_column, run_checks, ...)

\S4method{problem}{sf,character}(x, features, cost_column, ...)

\S4method{problem}{sf,ZonesCharacter}(x, features, cost_column, ...)
}
\arguments{
\item{x}{\code{\link[raster]{Raster-class}},
\code{\link[sf]{sf}},
\code{\link[sp]{SpatialPolygonsDataFrame-class}},
\code{\link[sp]{SpatialLinesDataFrame-class}},
\code{\link[sp]{SpatialPointsDataFrame-class}},
\code{\link{data.frame}} object,
\code{\link{numeric}} vector, or
\code{\link{matrix}} specifying the planning units to use in the reserve
design exercise and their corresponding cost. It may be desirable to
exclude some planning units from the analysis, for example those outside
the study area. To exclude planning units, set the cost for those raster
cells to \code{NA}, or use the \code{add_locked_out_constraint} function.}

\item{features}{The feature data can be specified in a variety of ways.
  The specific formats that can be used depend on the cost data format (i.e.
  argument to \code{x}) and whether the problem should have a single zone or
  multiple zones. If the problem should have a single zone, then the feature
  data can be specified following:

  \itemize{

  \item \code{x = \link[raster]{RasterLayer-class}}, or
    \code{x = \link[sp]{Spatial-class}},
    or \code{x = \link[sf]{sf}}:
    \code{y = \link[raster]{Raster-class}}
    object showing the distribution of conservation features. Missing
    values (i.e. \code{NA} values) can be used to indicate the absence of
    a feature in a particular cell instead of explicitly setting these
    cells to zero. Note that this argument type for \code{features} can
    only be used to specify data for problems involving a single zone.

  \item \code{x = \link[sp]{Spatial-class}},
    \code{x = \link[sf]{sf}},
    or \code{x = data.frame}:
    \code{y = character} vector
    with column names that correspond to the abundance or occurrence of
    different features in each planning unit. Note that this argument
    type can only be used to create problems involving a single zone.

  \item \code{x = \link{Spatial-class}},
    \code{x = \link[sf]{sf}},
    \code{x = data.frame},
    \code{x = numeric} vector, or
    \code{x = \link{matrix}}: \code{y = data.frame} object
    containing the names of the features. Note that if this
    type of argument is supplied to \code{features} then the argument
    \code{rij} or \code{rij_matrix} must also be supplied. This type of
    argument should follow the conventions used by \emph{Marxan}, wherein
    each row corresponds to a different feature. It must also contain the
    following columns:

    \describe{

    \item{\code{"id"}}{\code{integer} unique identifier for each feature
      These identifiers are used in the argument to \code{rij}.}
    \item{\code{"name"}}{\code{character} name for each feature.}
    \item{\code{"prop"}}{\code{numeric} relative target for each feature
      (optional).}
    \item{\code{"amount"}}{\code{numeric} absolute target for each
      feature (optional).}
    }
  }

  If the problem should have multiple zones, then the feature
  data can be specified following:

  \itemize{

  \item \code{x = \link[raster]{RasterStack-class}},
    \code{x = \link[raster]{RasterBrick-class}},
    \code{x = \link[sp]{Spatial-class}},
    or \code{x = \link[sf]{sf}}: \code{y = \link{ZonesRaster}}
    object showing the distribution of conservation features in multiple
    zones. As above, missing values (i.e. \code{NA} values) can be used to
    indicate the absence of a feature in a particular cell instead of
    explicitly setting these cells to zero.

  \item \code{x = \link[sp]{Spatial-class}},
    \code{x = \link[sf]{sf}},
    or \code{x = data.frame}: \code{y = \link{ZonesCharacter}}
    object with column names that correspond to the abundance or
    occurrence of different features in each planning unit in different
    zones.

 }}

\item{...}{not used.}

\item{run_checks}{\code{logical} flag indicating whether checks should be
run to ensure the integrity of the input data. These checks are run by
default; however, for large data sets they may increase run time. If it is
taking a prohibitively long time to create the prioritization problem, it
is suggested to try setting \code{run_checks} to \code{FALSE}.}

\item{cost_column}{\code{character} name or \code{integer} indicating the
column(s) with the cost data. This argument must be supplied when the
argument to \code{x} is a \code{\link[sp]{Spatial-class}} or
\code{data.frame} object. This argument should contain the name of each
column containing cost data for each management zone when creating
problems with multiple zones. To create a problem with a single zone, then
set the argument to \code{cost_column} as a single column name.}

\item{rij}{\code{data.frame} containing information on the amount of
   each feature in each planning unit assuming each management zone. Similar
   to \code{data.frame} arguments for \code{features}, the \code{data.frame}
   objects must follow the conventions used by \emph{Marxan}. Note that the
   \code{"zone"} column is not needed for problems involving a single
   management zone. Specifically, the argument should contain the following
   columns:

   \describe{

   \item{\code{"pu"}}{\code{integer} planning unit identifier.}

   \item{\code{"species"}}{\code{integer} feature identifier.}

   \item{\code{"zone"}}{\code{integer} zone identifier (optional for
     problems involving a single zone).}

   \item{\code{"amount"}}{\code{numeric} amount of the feature in the
     planning unit.}

   }}

\item{zones}{\code{data.frame} containing information on the zones. This
  argument is only used when argument to \code{x} and \code{y} are
  both \code{data.frame} objects and the problem being built contains
  multiple zones. Following conventions used in \code{MarZone}, this
  argument should contain the following columns:
  columns:

  \describe{

  \item{\code{"id"}}{\code{integer} zone identifier.}

  \item{\code{"name"}}{\code{character} zone name.}

  }}

\item{rij_matrix}{\code{list} of \code{matrix} or
\code{\link[Matrix]{dgCMatrix-class}}
objects specifying the amount of each feature (rows) within each planning
unit (columns) for each zone. The \code{list} elements denote
different zones, matrix rows denote features, and matrix columns denote
planning units. For convenience, the argument to
\code{rij_matrix} can be a single \code{matrix} or
\code{\link[Matrix]{dgCMatrix-class}} when specifying a problem with a
single management zone. This argument is only used when the argument
to \code{x} is a \code{numeric} or \code{matrix} object.}
}
\value{
A \code{\link{ConservationProblem-class}} object containing the
  basic data used to build a prioritization problem.
}
\description{
Create a systematic conservation planning problem. This function is used to
specify the basic data used in a spatial prioritization problem: the
spatial distribution of the planning units and their costs, as well as
the features (e.g. species, ecosystems) that need to be conserved. After
constructing this \code{ConservationProblem-class} object, it can be
customized to meet specific goals using \code{\link{objectives}},
\code{\link{targets}}, \code{\link{constraints}}, and
\code{\link{penalties}}. After building the problem, the
 \code{\link{solve}} function can be used to identify solutions.
}
\details{
A reserve design exercise starts by dividing the study region
  into planning units (typically square or hexagonal cells) and, for
  each planning unit, assigning values that quantify socioeconomic
  cost and conservation benefit for a set of conservation features. The
  cost can be the acquisition cost of the land, the cost of management,
  the opportunity cost of foregone commercial activities (e.g. from logging
  or agriculture), or simply the area. The conservation features are
  typically species (e.g. Clouded Leopard) or habitats (e.g. mangroves or
  cloud forest). The benefit that each feature derives from a planning unit
  can take a variety of forms, but is typically either occupancy (i.e.
  presence or absence) or area of occurrence within each planning unit.
  Finally, in some types of reserve design models, representation targets
  must be set for each conservation feature, such as 20 % of the current
  extent of cloud forest or 10,000 km^2 of Clouded Leopard habitat
  (see \code{\link{targets}}).

  The goal of the reserve design exercise is then to optimize the trade-off
  between conservation benefit and socioeconomic cost, i.e. to get the most
  benefit for your limited conservation funds. In general, the goal of an
  optimization problem is to minimize an objective function over a set of
  decision variables, subject to a series of constraints. The decision
  variables are what we control, usually there is one binary variable for
  each planning unit specifying whether or not to protect that unit (but
  other approaches are available, see \code{\link{decisions}}). The
  constraints can be thought of as rules that need to be followed, for
  example, that the reserve must stay within a certain budget or meet the
  representation targets.

  Integer linear programming (ILP) is the subset of optimization algorithms
  used in this package to solve reserve design problems. The general form of
  an integer programming problem can be expressed in matrix notation using
  the following equation.

  \deqn{\mathit{Minimize} \space \mathbf{c}^{\mathbf{T}}\mathbf{x} \space
  \mathit{subject \space to} \space
  \mathbf{Ax}\geq= or\leq \mathbf{b}}{Minimize (c^T)*x subject to Ax \ge, =,
  or \le b}

  Here, \emph{x} is a vector of decision variables, \emph{c} and \emph{b} are
  vectors of known coefficients, and \emph{A} is the constraint
  matrix. The final term specifies a series of structural
  constraints where relational operators for the constraint can be either
  \eqn{\ge}, \eqn{=}, or \eqn{\le} the coefficients. For example, in the
  minimum set
  cover problem, \emph{c} would be a vector of costs for each planning unit,
  \emph{b} a vector of targets for each conservation feature, the relational
  operator would be \eqn{\ge} for all features, and \emph{A} would be the
  representation matrix with \eqn{A_{ij}=r_{ij}}{Aij = rij}, the
  representation level of feature \emph{i} in planning unit \emph{j}.

  Please note that this function internally computes the amount of each
  feature in each planning unit when this data is not supplied (using the
  \code{rij_matrix} parameter). As a consequence, it can take a while to
  initialize large-scale conservation planning problems that involve
  millions of planning units.
}
\examples{
# load data
data(sim_pu_raster, sim_pu_polygons, sim_pu_lines, sim_pu_points,
     sim_pu_sf, sim_features)

# create problem using raster planning unit data
p1 <- problem(sim_pu_raster, sim_features) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

\donttest{
# create problem using polygon (Spatial) planning unit data
p2 <- problem(sim_pu_polygons, sim_features, "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# create problem using line (Spatial) planning unit data
p3 <- problem(sim_pu_lines, sim_features, "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# create problem using point (Spatial) planning unit data
p4 <- problem(sim_pu_points, sim_features, "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# create problem using polygon (sf) planning unit data
p5 <- problem(sim_pu_sf, sim_features, "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# add columns to polygon planning unit data representing the abundance
# of species inside them
sim_pu_polygons$spp_1 <- rpois(length(sim_pu_polygons), 5)
sim_pu_polygons$spp_2 <- rpois(length(sim_pu_polygons), 8)
sim_pu_polygons$spp_3 <- rpois(length(sim_pu_polygons), 2)

# create problem using pre-processed data when feature abundances are
# stored in the columns of an attribute table for a spatial vector data set
p6 <- problem(sim_pu_polygons, features = c("spp_1", "spp_2", "spp_3"),
              "cost") \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# alternatively one can supply pre-processed aspatial data
costs <- sim_pu_polygons$cost
features <- data.frame(id = seq_len(nlayers(sim_features)),
                       name = names(sim_features))
rij_mat <- rij_matrix(sim_pu_polygons, sim_features)
p7 <- problem(costs, features, rij_matrix = rij_mat) \%>\%
      add_min_set_objective() \%>\%
      add_relative_targets(0.2) \%>\%
      add_binary_decisions()

# solve problems
s1 <- solve(p1)
s2 <- solve(p2)
s3 <- solve(p3)
s4 <- solve(p4)
s5 <- solve(p5)
s6 <- solve(p6)
s7 <- solve(p7)

# plot solutions for problems associated with spatial data
par(mfrow = c(3, 2), mar = c(0, 0, 4.1, 0))
plot(s1, main = "raster data", axes = FALSE, box = FALSE)

plot(s2, main = "polygon data")
plot(s2[s2$solution_1 == 1, ], col = "darkgreen", add = TRUE)

plot(s3, main = "line data")
lines(s3[s3$solution_1 == 1, ], col = "darkgreen", lwd = 2)

plot(s4, main = "point data", pch = 19)
points(s4[s4$solution_1 == 1, ], col = "darkgreen", cex = 2, pch = 19)

plot(s5, main = "sf (polygon) data", pch = 19)
points(s5[s5$solution_1 == 1, ], col = "darkgreen", cex = 2, pch = 19)

plot(s6, main = "preprocessed data", pch = 19)
plot(s6[s6$solution_1 == 1, ], col = "darkgreen", add = TRUE)

# show solutions for problems associated with aspatial data
str(s7)
}
# create some problems with multiple zones

# first, create a matrix containing the targets for multi-zone problems
# here each row corresponds to a different feature, each
# column corresponds to a different zone, and values correspond
# to the total (absolute) amount of a given feature that needs to be secured
# in a given zone
targets <- matrix(rpois(15, 1),
                  nrow = number_of_features(sim_features_zones),
                  ncol = number_of_zones(sim_features_zones),
                  dimnames = list(feature_names(sim_features_zones),
                                  zone_names(sim_features_zones)))

# print targets
print(targets)

# create a multi-zone problem with raster data
p8 <- problem(sim_pu_zones_stack, sim_features_zones) \%>\%
      add_min_set_objective() \%>\%
      add_absolute_targets(targets) \%>\%
      add_binary_decisions()
\donttest{
# solve problem
s8 <- solve(p8)

# plot solution
# here, each layer/panel corresponds to a different zone and pixel values
# indicate if a given planning unit has been allocated to a given zone
par(mfrow = c(1, 1))
plot(s8, main = c("zone 1", "zone 2", "zone 3"), axes = FALSE, box = FALSE)

# alternatively, the category_layer function can be used to create
# a new raster object containing the zone ids for each planning unit
# in the solution (note this only works for problems with binary decisions)
par(mfrow = c(1, 1))
plot(category_layer(s8), axes = FALSE, box = FALSE)

# create a multi-zone problem with polygon data
p9 <- problem(sim_pu_zones_polygons, sim_features_zones,
              cost_column = c("cost_1", "cost_2", "cost_3")) \%>\%
      add_min_set_objective() \%>\%
      add_absolute_targets(targets) \%>\%
      add_binary_decisions()

# solve problem
s9 <- solve(p9)

# create column containing the zone id for which each planning unit was
# allocated to in the solution
s9$solution <- category_vector(s9@data[, c("solution_1_zone_1",
                                           "solution_1_zone_2",
                                           "solution_1_zone_3")])
s9$solution <- factor(s9$solution)

# plot solution
spplot(s9, zcol = "solution", main = "solution", axes = FALSE, box = FALSE)

# create a multi-zone problem with polygon planning unit data
# and where fields (columns) in the attribute table correspond
# to feature abundances

# first fields need to be added to the planning unit data
# which indicate the amount of each feature in each zone
# to do this, the fields will be populated with random counts
sim_pu_zones_polygons$spp1_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
sim_pu_zones_polygons$spp2_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
sim_pu_zones_polygons$spp3_z1 <- rpois(nrow(sim_pu_zones_polygons), 1)
sim_pu_zones_polygons$spp1_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)
sim_pu_zones_polygons$spp2_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)
sim_pu_zones_polygons$spp3_z2 <- rpois(nrow(sim_pu_zones_polygons), 1)

# create problem with polygon planning unit data and use field names
# to indicate feature data
# additionally, to make this example slightly more interesting,
# the problem will have prfoportion-type decisions such that
# a proportion of each planning unit can be allocated to each of the
# two management zones
p10 <- problem(sim_pu_zones_polygons,
               zones(c("spp1_z1", "spp2_z1", "spp3_z1"),
                     c("spp1_z2", "spp2_z2", "spp3_z2"),
                     zone_names = c("z1", "z2")),
               cost_column = c("cost_1", "cost_2")) \%>\%
       add_min_set_objective() \%>\%
       add_absolute_targets(targets[1:3, 1:2]) \%>\%
       add_proportion_decisions()

# solve problem
s10 <- solve(p10)

# plot solution
spplot(s10, zcol = c("solution_1_z1", "solution_1_z2"), main = "solution",
       axes = FALSE, box = FALSE)
}
}
\seealso{
\code{\link{constraints}}, \code{\link{decisions}},
 \code{\link{objectives}} \code{\link{penalties}},
 \code{\link{portfolios}}, \code{\link{solvers}}, \code{\link{targets}},
 \code{\link{feature_representation}}, \code{\link{irreplaceability}}.
}
