/*
 * Decompiled with CFR 0.152.
 */
package org.openscience.cdk.renderer.generators.standard;

import com.google.common.primitives.Ints;
import java.awt.Color;
import java.awt.Font;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.vecmath.Point2d;
import javax.vecmath.Tuple2d;
import javax.vecmath.Vector2d;
import org.openscience.cdk.config.Elements;
import org.openscience.cdk.graph.Cycles;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IPseudoAtom;
import org.openscience.cdk.interfaces.IRingSet;
import org.openscience.cdk.renderer.RendererModel;
import org.openscience.cdk.renderer.color.IAtomColorer;
import org.openscience.cdk.renderer.elements.ElementGroup;
import org.openscience.cdk.renderer.elements.GeneralPath;
import org.openscience.cdk.renderer.elements.IRenderingElement;
import org.openscience.cdk.renderer.elements.LineElement;
import org.openscience.cdk.renderer.elements.MarkedElement;
import org.openscience.cdk.renderer.elements.path.Close;
import org.openscience.cdk.renderer.elements.path.CubicTo;
import org.openscience.cdk.renderer.elements.path.LineTo;
import org.openscience.cdk.renderer.elements.path.MoveTo;
import org.openscience.cdk.renderer.elements.path.PathElement;
import org.openscience.cdk.renderer.generators.BasicSceneGenerator;
import org.openscience.cdk.renderer.generators.standard.AtomSymbol;
import org.openscience.cdk.renderer.generators.standard.StandardGenerator;
import org.openscience.cdk.renderer.generators.standard.TextOutline;
import org.openscience.cdk.renderer.generators.standard.VecmathUtil;
import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.openscience.cdk.tools.manipulator.AtomContainerSetManipulator;

final class StandardBondGenerator {
    private final IAtomContainer container;
    private final AtomSymbol[] symbols;
    private final RendererModel parameters;
    private final ILoggingTool logger = LoggingToolFactory.createLoggingTool(this.getClass());
    private final Map<IAtom, Integer> atomIndexMap = new HashMap<IAtom, Integer>();
    private final Map<IBond, IAtomContainer> ringMap;
    private final double scale;
    private final double stroke;
    private final double separation;
    private final double backOff;
    private final double wedgeWidth;
    private final double hashSpacing;
    private final double waveSpacing;
    private final Color foreground;
    private final Color annotationColor;
    private final boolean fancyBoldWedges;
    private final boolean fancyHashedWedges;
    private final double annotationDistance;
    private final double annotationScale;
    private final Font font;
    private final ElementGroup annotations;

    private StandardBondGenerator(IAtomContainer container, AtomSymbol[] symbols, RendererModel parameters, ElementGroup annotations, Font font, double stroke) {
        this.container = container;
        this.symbols = symbols;
        this.parameters = parameters;
        this.annotations = annotations;
        for (int i = 0; i < container.getAtomCount(); ++i) {
            this.atomIndexMap.put(container.getAtom(i), i);
        }
        this.ringMap = StandardBondGenerator.ringPreferenceMap(container);
        this.scale = (Double)parameters.get(BasicSceneGenerator.Scale.class);
        this.stroke = stroke;
        double length = (Double)parameters.get(BasicSceneGenerator.BondLength.class) / this.scale;
        this.separation = (Double)parameters.get(StandardGenerator.BondSeparation.class) * (Double)parameters.get(BasicSceneGenerator.BondLength.class) / this.scale;
        this.backOff = (Double)parameters.get(StandardGenerator.SymbolMarginRatio.class) * stroke;
        this.wedgeWidth = (Double)parameters.get(StandardGenerator.WedgeRatio.class) * stroke;
        this.hashSpacing = (Double)parameters.get(StandardGenerator.HashSpacing.class) / this.scale;
        this.waveSpacing = (Double)parameters.get(StandardGenerator.WaveSpacing.class) / this.scale;
        this.fancyBoldWedges = (Boolean)parameters.get(StandardGenerator.FancyBoldWedges.class);
        this.fancyHashedWedges = (Boolean)parameters.get(StandardGenerator.FancyHashedWedges.class);
        this.annotationDistance = (Double)parameters.get(StandardGenerator.AnnotationDistance.class) * ((Double)parameters.get(BasicSceneGenerator.BondLength.class) / this.scale);
        this.annotationScale = 1.0 / this.scale * (Double)parameters.get(StandardGenerator.AnnotationFontScale.class);
        this.annotationColor = (Color)parameters.get(StandardGenerator.AnnotationColor.class);
        this.font = font;
        this.foreground = ((IAtomColorer)parameters.get(StandardGenerator.AtomColor.class)).getAtomColor(container.getBuilder().newInstance(IAtom.class, "C"));
    }

    static IRenderingElement[] generateBonds(IAtomContainer container, AtomSymbol[] symbols, RendererModel parameters, double stroke, Font font, ElementGroup annotations) {
        StandardBondGenerator bondGenerator = new StandardBondGenerator(container, symbols, parameters, annotations, font, stroke);
        IRenderingElement[] elements = new IRenderingElement[container.getBondCount()];
        for (int i = 0; i < container.getBondCount(); ++i) {
            IBond bond = container.getBond(i);
            if (StandardGenerator.isHidden(bond)) continue;
            elements[i] = bondGenerator.generate(bond);
        }
        return elements;
    }

    IRenderingElement generate(IBond bond) {
        ElementGroup elemGrp;
        IRenderingElement elem;
        IAtom atom1 = bond.getAtom(0);
        IAtom atom2 = bond.getAtom(1);
        IBond.Order order = bond.getOrder();
        if (order == null) {
            order = IBond.Order.UNSET;
        }
        switch (order) {
            case SINGLE: {
                elem = this.generateSingleBond(bond, atom1, atom2);
                break;
            }
            case DOUBLE: {
                elem = this.generateDoubleBond(bond);
                break;
            }
            case TRIPLE: {
                elem = this.generateTripleBond(bond, atom1, atom2);
                break;
            }
            default: {
                elem = this.generateDashedBond(atom1, atom2);
            }
        }
        if (this.isAttachPoint(atom1)) {
            elemGrp = new ElementGroup();
            elemGrp.add(elem);
            elemGrp.add(this.generateAttachPoint(atom1, bond));
            elem = elemGrp;
        }
        if (this.isAttachPoint(atom2)) {
            elemGrp = new ElementGroup();
            elemGrp.add(elem);
            elemGrp.add(this.generateAttachPoint(atom2, bond));
            elem = elemGrp;
        }
        return elem;
    }

    private IRenderingElement generateSingleBond(IBond bond, IAtom from, IAtom to) {
        IBond.Stereo stereo = bond.getStereo();
        if (stereo == null) {
            return this.generatePlainSingleBond(from, to);
        }
        List<IBond> fromBonds = this.container.getConnectedBondsList(from);
        List<IBond> toBonds = this.container.getConnectedBondsList(to);
        fromBonds.remove(bond);
        toBonds.remove(bond);
        String label = StandardGenerator.getAnnotationLabel(bond);
        if (label != null) {
            this.addAnnotation(from, to, label);
        }
        switch (stereo) {
            case NONE: {
                return this.generatePlainSingleBond(from, to);
            }
            case DOWN: {
                return this.generateHashedWedgeBond(from, to, toBonds);
            }
            case DOWN_INVERTED: {
                return this.generateHashedWedgeBond(to, from, fromBonds);
            }
            case UP: {
                return this.generateBoldWedgeBond(from, to, toBonds);
            }
            case UP_INVERTED: {
                return this.generateBoldWedgeBond(to, from, fromBonds);
            }
            case UP_OR_DOWN: 
            case UP_OR_DOWN_INVERTED: {
                return this.generateWavyBond(to, from);
            }
        }
        this.logger.warn("Unknown single bond stereochemistry ", new Object[]{stereo, " is not displayed"});
        return this.generatePlainSingleBond(from, to);
    }

    IRenderingElement generatePlainSingleBond(IAtom from, IAtom to) {
        return this.newLineElement(this.backOffPoint(from, to), this.backOffPoint(to, from));
    }

    IRenderingElement generateBoldWedgeBond(IAtom from, IAtom to, List<IBond> toBonds) {
        Point2d fromPoint = from.getPoint2d();
        Point2d toPoint = to.getPoint2d();
        Point2d fromBackOffPoint = this.backOffPoint(from, to);
        Point2d toBackOffPoint = this.backOffPoint(to, from);
        Vector2d unit = VecmathUtil.newUnitVector(fromPoint, toPoint);
        Vector2d perpendicular = VecmathUtil.newPerpendicularVector(unit);
        double halfNarrowEnd = this.stroke / 2.0;
        double halfWideEnd = this.wedgeWidth / 2.0;
        double opposite = halfWideEnd - halfNarrowEnd;
        double adjacent = fromPoint.distance(toPoint);
        double fromOffset = halfNarrowEnd + opposite / adjacent * fromBackOffPoint.distance(fromPoint);
        double toOffset = halfNarrowEnd + opposite / adjacent * toBackOffPoint.distance(fromPoint);
        Vector2d a = VecmathUtil.sum(fromBackOffPoint, VecmathUtil.scale(perpendicular, fromOffset));
        Vector2d b = VecmathUtil.sum(fromBackOffPoint, VecmathUtil.scale(perpendicular, -fromOffset));
        Tuple2d c = VecmathUtil.sum(toBackOffPoint, VecmathUtil.scale(perpendicular, -toOffset));
        Tuple2d e = toBackOffPoint;
        Tuple2d d = VecmathUtil.sum(toBackOffPoint, VecmathUtil.scale(perpendicular, toOffset));
        double threshold = Math.toRadians(15.0);
        if (this.fancyBoldWedges && !this.hasDisplayedSymbol(to)) {
            if (toBonds.size() == 1) {
                double theta;
                IBond toBondNeighbor = toBonds.get(0);
                IAtom toNeighbor = toBondNeighbor.getConnectedAtom(to);
                Vector2d refVector = VecmathUtil.newUnitVector(toPoint, toNeighbor.getPoint2d());
                boolean wideToWide = false;
                if (this.atWideEndOfWedge(to, toBondNeighbor)) {
                    refVector = VecmathUtil.sum(refVector, VecmathUtil.negate(unit));
                    wideToWide = true;
                }
                if ((theta = refVector.angle(unit)) > threshold) {
                    c = VecmathUtil.intersection(b, VecmathUtil.newUnitVector(b, c), toPoint, refVector);
                    d = VecmathUtil.intersection(a, VecmathUtil.newUnitVector(a, d), toPoint, refVector);
                    if (!wideToWide) {
                        double nudge = this.stroke / 2.0 / Math.sin(theta);
                        c = VecmathUtil.sum(c, VecmathUtil.scale(unit, nudge));
                        d = VecmathUtil.sum(d, VecmathUtil.scale(unit, nudge));
                        e = VecmathUtil.sum(e, VecmathUtil.scale(unit, nudge));
                    }
                }
            } else if (toBonds.size() > 1) {
                Vector2d refVectorA = VecmathUtil.getNearestVector(perpendicular, to, toBonds);
                Vector2d refVectorB = VecmathUtil.getNearestVector(VecmathUtil.negate(perpendicular), to, toBonds);
                if (refVectorB.angle(unit) > threshold) {
                    c = VecmathUtil.intersection(b, VecmathUtil.newUnitVector(b, c), toPoint, refVectorB);
                }
                if (refVectorA.angle(unit) > threshold) {
                    d = VecmathUtil.intersection(a, VecmathUtil.newUnitVector(a, d), toPoint, refVectorA);
                }
            }
        }
        return new GeneralPath(Arrays.asList(new MoveTo(new Point2d(a)), new LineTo(new Point2d(b)), new LineTo(new Point2d(c)), new LineTo(new Point2d(e)), new LineTo(new Point2d(d)), new Close()), this.foreground);
    }

    IRenderingElement generateHashedWedgeBond(IAtom from, IAtom to, List<IBond> toBonds) {
        Point2d fromPoint = from.getPoint2d();
        Point2d toPoint = to.getPoint2d();
        Point2d fromBackOffPoint = this.backOffPoint(from, to);
        Point2d toBackOffPoint = this.backOffPoint(to, from);
        Vector2d unit = VecmathUtil.newUnitVector(fromPoint, toPoint);
        Vector2d perpendicular = VecmathUtil.newPerpendicularVector(unit);
        double halfNarrowEnd = this.stroke / 2.0;
        double halfWideEnd = this.wedgeWidth / 2.0;
        double opposite = halfWideEnd - halfNarrowEnd;
        double adjacent = fromPoint.distance(toPoint);
        int nSections = (int)(adjacent / this.hashSpacing);
        double step = adjacent / (double)(nSections - 1);
        ElementGroup group = new ElementGroup();
        double start = this.hasDisplayedSymbol(from) ? fromPoint.distance(fromBackOffPoint) : Double.NEGATIVE_INFINITY;
        double end = this.hasDisplayedSymbol(to) ? fromPoint.distance(toBackOffPoint) : Double.POSITIVE_INFINITY;
        double threshold = Math.toRadians(35.0);
        Vector2d hatchAngle = perpendicular;
        if (this.canDrawFancyHashedWedge(to, toBonds, adjacent)) {
            IBond toBondNeighbor = toBonds.get(0);
            IAtom toNeighbor = toBondNeighbor.getConnectedAtom(to);
            Vector2d refVector = VecmathUtil.newUnitVector(toPoint, toNeighbor.getPoint2d());
            if (this.atWideEndOfWedge(to, toBondNeighbor)) {
                refVector = VecmathUtil.sum(refVector, VecmathUtil.negate(unit));
                refVector.normalize();
            }
            if (refVector.angle(unit) > threshold) {
                hatchAngle = refVector;
            }
        }
        for (int i = 0; i < nSections; ++i) {
            double distance = (double)i * step;
            if (distance < start || distance > end) continue;
            double offset = halfNarrowEnd + opposite / adjacent * distance;
            Vector2d interval = VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, distance));
            group.add(this.newLineElement(VecmathUtil.sum(interval, VecmathUtil.scale(hatchAngle, offset)), VecmathUtil.sum(interval, VecmathUtil.scale(hatchAngle, -offset))));
        }
        return group;
    }

    private boolean canDrawFancyHashedWedge(IAtom to, List<IBond> toBonds, double length) {
        boolean longBond = length * this.scale - (Double)this.parameters.get(BasicSceneGenerator.BondLength.class) > 4.0;
        return this.fancyHashedWedges && !longBond && !this.hasDisplayedSymbol(to) && toBonds.size() == 1;
    }

    IRenderingElement generateWavyBond(IAtom from, IAtom to) {
        Point2d fromPoint = from.getPoint2d();
        Point2d toPoint = to.getPoint2d();
        Point2d fromBackOffPoint = this.backOffPoint(from, to);
        Point2d toBackOffPoint = this.backOffPoint(to, from);
        Vector2d unit = VecmathUtil.newUnitVector(fromPoint, toPoint);
        Vector2d perpendicular = VecmathUtil.newPerpendicularVector(unit);
        double length = fromPoint.distance(toPoint);
        int nCurves = 2 * (int)(length / this.waveSpacing);
        double step = length / (double)nCurves;
        Vector2d peak = VecmathUtil.scale(perpendicular, step);
        boolean started = false;
        double start = fromPoint.equals(fromBackOffPoint) ? Double.MIN_VALUE : fromPoint.distance(fromBackOffPoint);
        double end = toPoint.equals(toBackOffPoint) ? Double.MAX_VALUE : fromPoint.distance(toBackOffPoint);
        ArrayList<PathElement> path = new ArrayList<PathElement>();
        if (start == Double.MIN_VALUE) {
            path.add(new MoveTo(fromPoint.x, fromPoint.y));
            started = true;
        }
        for (int i = 1; i < nCurves; i += 2) {
            Vector2d controlPoint2;
            Vector2d controlPoint1;
            Vector2d endPoint;
            peak = VecmathUtil.negate(peak);
            double dist = (double)i * step;
            if (dist >= start && dist <= end) {
                endPoint = VecmathUtil.sum(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, dist)), peak);
                if (started) {
                    controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, (double)(i - 1) * step)), VecmathUtil.scale(peak, 0.5));
                    controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, ((double)i - 0.5) * step)), peak);
                    path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
                } else {
                    path.add(new MoveTo(endPoint.x, endPoint.y));
                    started = true;
                }
            }
            if (!((dist = (double)(i + 1) * step) >= start) || !(dist <= end)) continue;
            endPoint = VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, dist));
            if (started) {
                controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, ((double)i + 0.5) * step)), peak);
                controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, dist)), VecmathUtil.scale(peak, 0.5));
                path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
                continue;
            }
            path.add(new MoveTo(endPoint.x, endPoint.y));
            started = true;
        }
        return new GeneralPath(path, this.foreground).outline(this.stroke);
    }

    private IRenderingElement generateDoubleBond(IBond bond) {
        boolean cyclic = this.ringMap.containsKey(bond);
        IAtomContainer refContainer = cyclic ? this.ringMap.get(bond) : this.container;
        int length = refContainer.getAtomCount();
        int index1 = refContainer.getAtomNumber(bond.getAtom(0));
        int index2 = refContainer.getAtomNumber(bond.getAtom(1));
        boolean outOfOrder = cyclic && index1 == (index2 + 1) % length;
        IAtom atom1 = bond.getAtom(outOfOrder ? 1 : 0);
        IAtom atom2 = bond.getAtom(outOfOrder ? 0 : 1);
        if (IBond.Stereo.E_OR_Z.equals((Object)bond.getStereo())) {
            return this.generateCrossedDoubleBond(atom1, atom2);
        }
        List<IBond> atom1Bonds = refContainer.getConnectedBondsList(atom1);
        List<IBond> atom2Bonds = refContainer.getConnectedBondsList(atom2);
        atom1Bonds.remove(bond);
        atom2Bonds.remove(bond);
        if (cyclic) {
            int wind1 = StandardBondGenerator.winding(atom1Bonds.get(0), bond);
            int wind2 = StandardBondGenerator.winding(bond, atom2Bonds.get(0));
            if (wind1 > 0 && !this.hasDisplayedSymbol(atom1)) {
                return this.generateOffsetDoubleBond(bond, atom1, atom2, atom1Bonds.get(0), atom2Bonds);
            }
            if (wind2 > 0 && !this.hasDisplayedSymbol(atom2)) {
                return this.generateOffsetDoubleBond(bond, atom2, atom1, atom2Bonds.get(0), atom1Bonds);
            }
            if (!this.hasDisplayedSymbol(atom1)) {
                return this.generateOffsetDoubleBond(bond, atom1, atom2, atom1Bonds.get(0), atom2Bonds, true);
            }
            if (!this.hasDisplayedSymbol(atom2)) {
                return this.generateOffsetDoubleBond(bond, atom2, atom1, atom2Bonds.get(0), atom1Bonds, true);
            }
            return this.generateCenteredDoubleBond(bond, atom1, atom2, atom1Bonds, atom2Bonds);
        }
        if (!(atom1Bonds.size() != 1 || this.hasDisplayedSymbol(atom1) || this.hasDisplayedSymbol(atom2) && !atom2Bonds.isEmpty())) {
            return this.generateOffsetDoubleBond(bond, atom1, atom2, atom1Bonds.get(0), atom2Bonds);
        }
        if (!(atom2Bonds.size() != 1 || this.hasDisplayedSymbol(atom2) || this.hasDisplayedSymbol(atom1) && !atom1Bonds.isEmpty())) {
            return this.generateOffsetDoubleBond(bond, atom2, atom1, atom2Bonds.get(0), atom1Bonds);
        }
        if (this.specialOffsetBondNextToWedge(atom1, atom1Bonds) && !this.hasDisplayedSymbol(atom1)) {
            return this.generateOffsetDoubleBond(bond, atom1, atom2, this.selectPlainSingleBond(atom1Bonds), atom2Bonds);
        }
        if (this.specialOffsetBondNextToWedge(atom2, atom2Bonds) && !this.hasDisplayedSymbol(atom2)) {
            return this.generateOffsetDoubleBond(bond, atom2, atom1, this.selectPlainSingleBond(atom2Bonds), atom1Bonds);
        }
        return this.generateCenteredDoubleBond(bond, atom1, atom2, atom1Bonds, atom2Bonds);
    }

    private boolean specialOffsetBondNextToWedge(IAtom atom, List<IBond> bonds) {
        if (bonds.size() != 2) {
            return false;
        }
        if (this.atWideEndOfWedge(atom, bonds.get(0)) && StandardBondGenerator.isPlainBond(bonds.get(1))) {
            return true;
        }
        return this.atWideEndOfWedge(atom, bonds.get(1)) && StandardBondGenerator.isPlainBond(bonds.get(0));
    }

    private IBond selectPlainSingleBond(List<IBond> bonds) {
        for (IBond bond : bonds) {
            if (!StandardBondGenerator.isPlainBond(bond)) continue;
            return bond;
        }
        return bonds.get(0);
    }

    private static boolean isPlainBond(IBond bond) {
        return IBond.Order.SINGLE.equals((Object)bond.getOrder()) && (bond.getStereo() == null || bond.getStereo() == IBond.Stereo.NONE);
    }

    private boolean atWideEndOfWedge(IAtom atom, IBond bond) {
        if (bond.getStereo() == null) {
            return false;
        }
        switch (bond.getStereo()) {
            case UP: {
                return bond.getAtom(1) == atom;
            }
            case UP_INVERTED: {
                return bond.getAtom(0) == atom;
            }
            case DOWN: {
                return bond.getAtom(1) == atom;
            }
            case DOWN_INVERTED: {
                return bond.getAtom(0) == atom;
            }
        }
        return false;
    }

    private IRenderingElement generateOffsetDoubleBond(IBond bond, IAtom atom1, IAtom atom2, IBond atom1Bond, List<IBond> atom2Bonds) {
        return this.generateOffsetDoubleBond(bond, atom1, atom2, atom1Bond, atom2Bonds, false);
    }

    private IRenderingElement generateOffsetDoubleBond(IBond bond, IAtom atom1, IAtom atom2, IBond atom1Bond, List<IBond> atom2Bonds, boolean invert) {
        double halfBondLength;
        assert (!this.hasDisplayedSymbol(atom1));
        assert (atom1Bond != null);
        Point2d atom1Point = atom1.getPoint2d();
        Point2d atom2Point = atom2.getPoint2d();
        Point2d atom2BackOffPoint = this.backOffPoint(atom2, atom1);
        Vector2d unit = VecmathUtil.newUnitVector(atom1Point, atom2Point);
        Vector2d perpendicular = VecmathUtil.newPerpendicularVector(unit);
        Vector2d reference = VecmathUtil.newUnitVector(atom1.getPoint2d(), atom1Bond.getConnectedAtom(atom1).getPoint2d());
        if (reference.dot(perpendicular) < 0.0) {
            perpendicular = VecmathUtil.negate(perpendicular);
        }
        if (invert) {
            perpendicular = VecmathUtil.negate(perpendicular);
        }
        if (atom2Bonds.isEmpty() && this.hasDisplayedSymbol(atom2)) {
            int atom2index = this.atomIndexMap.get(atom2);
            Vector2d nudge = VecmathUtil.scale(perpendicular, this.separation / 2.0);
            this.symbols[atom2index] = this.symbols[atom2index].translate(nudge.x, nudge.y);
        }
        double atom1Offset = VecmathUtil.adjacentLength(VecmathUtil.sum(reference, unit), perpendicular, this.separation);
        double atom2Offset = 0.0;
        if (reference.dot(perpendicular) < 0.0) {
            atom1Offset = -atom1Offset;
        }
        if (!atom2Bonds.isEmpty() && !this.hasDisplayedSymbol(atom2)) {
            Vector2d closest = VecmathUtil.getNearestVector(perpendicular, atom2, atom2Bonds);
            atom2Offset = VecmathUtil.adjacentLength(VecmathUtil.sum(closest, VecmathUtil.negate(unit)), perpendicular, this.separation);
            if (closest.dot(perpendicular) < 0.0) {
                atom2Offset = -atom2Offset;
            }
        }
        if (atom1Offset > (halfBondLength = atom1Point.distance(atom2BackOffPoint) / 2.0) || atom1Offset < 0.0) {
            atom1Offset = 0.0;
        }
        if (atom2Offset > halfBondLength || atom2Offset < 0.0) {
            atom2Offset = 0.0;
        }
        ElementGroup group = new ElementGroup();
        group.add(this.newLineElement(atom1Point, atom2BackOffPoint));
        group.add(this.newLineElement(VecmathUtil.sum(VecmathUtil.sum(atom1Point, VecmathUtil.scale(perpendicular, this.separation)), VecmathUtil.scale(unit, atom1Offset)), VecmathUtil.sum(VecmathUtil.sum(atom2BackOffPoint, VecmathUtil.scale(perpendicular, this.separation)), VecmathUtil.scale(unit, -atom2Offset))));
        String label = StandardGenerator.getAnnotationLabel(bond);
        if (label != null) {
            this.addAnnotation(atom1, atom2, label, VecmathUtil.negate(perpendicular));
        }
        return group;
    }

    private IRenderingElement generateCenteredDoubleBond(IBond bond, IAtom atom1, IAtom atom2, List<IBond> atom1Bonds, List<IBond> atom2Bonds) {
        double line2Adjust;
        double line1Adjust;
        Vector2d nearest2;
        Vector2d nearest1;
        Point2d atom1BackOffPoint = this.backOffPoint(atom1, atom2);
        Point2d atom2BackOffPoint = this.backOffPoint(atom2, atom1);
        Vector2d unit = VecmathUtil.newUnitVector(atom1BackOffPoint, atom2BackOffPoint);
        Vector2d perpendicular1 = VecmathUtil.newPerpendicularVector(unit);
        Vector2d perpendicular2 = VecmathUtil.negate(perpendicular1);
        double halfBondLength = atom1BackOffPoint.distance(atom2BackOffPoint) / 2.0;
        double halfSeparation = this.separation / 2.0;
        ElementGroup group = new ElementGroup();
        Vector2d line1Atom1Point = VecmathUtil.sum(atom1BackOffPoint, VecmathUtil.scale(perpendicular1, halfSeparation));
        Vector2d line1Atom2Point = VecmathUtil.sum(atom2BackOffPoint, VecmathUtil.scale(perpendicular1, halfSeparation));
        Vector2d line2Atom1Point = VecmathUtil.sum(atom1BackOffPoint, VecmathUtil.scale(perpendicular2, halfSeparation));
        Vector2d line2Atom2Point = VecmathUtil.sum(atom2BackOffPoint, VecmathUtil.scale(perpendicular2, halfSeparation));
        if (!this.hasDisplayedSymbol(atom1) && atom1Bonds.size() > 1) {
            nearest1 = VecmathUtil.getNearestVector(perpendicular1, atom1, atom1Bonds);
            nearest2 = VecmathUtil.getNearestVector(perpendicular2, atom1, atom1Bonds);
            line1Adjust = VecmathUtil.adjacentLength(nearest1, perpendicular1, halfSeparation);
            line2Adjust = VecmathUtil.adjacentLength(nearest2, perpendicular2, halfSeparation);
            if (line1Adjust > halfBondLength || line1Adjust < 0.0) {
                line1Adjust = 0.0;
            }
            if (line2Adjust > halfBondLength || line2Adjust < 0.0) {
                line2Adjust = 0.0;
            }
            if (nearest1.dot(unit) > 0.0) {
                line1Adjust = -line1Adjust;
            }
            if (nearest2.dot(unit) > 0.0) {
                line2Adjust = -line2Adjust;
            }
            line1Atom1Point = VecmathUtil.sum(line1Atom1Point, VecmathUtil.scale(unit, -line1Adjust));
            line2Atom1Point = VecmathUtil.sum(line2Atom1Point, VecmathUtil.scale(unit, -line2Adjust));
        }
        if (!this.hasDisplayedSymbol(atom2) && atom2Bonds.size() > 1) {
            nearest1 = VecmathUtil.getNearestVector(perpendicular1, atom2, atom2Bonds);
            nearest2 = VecmathUtil.getNearestVector(perpendicular2, atom2, atom2Bonds);
            line1Adjust = VecmathUtil.adjacentLength(nearest1, perpendicular1, halfSeparation);
            line2Adjust = VecmathUtil.adjacentLength(nearest2, perpendicular2, halfSeparation);
            if (line1Adjust > halfBondLength || line1Adjust < 0.0) {
                line1Adjust = 0.0;
            }
            if (line2Adjust > halfBondLength || line2Adjust < 0.0) {
                line2Adjust = 0.0;
            }
            if (nearest1.dot(unit) < 0.0) {
                line1Adjust = -line1Adjust;
            }
            if (nearest2.dot(unit) < 0.0) {
                line2Adjust = -line2Adjust;
            }
            line1Atom2Point = VecmathUtil.sum(line1Atom2Point, VecmathUtil.scale(unit, line1Adjust));
            line2Atom2Point = VecmathUtil.sum(line2Atom2Point, VecmathUtil.scale(unit, line2Adjust));
        }
        group.add(this.newLineElement(line1Atom1Point, line1Atom2Point));
        group.add(this.newLineElement(line2Atom1Point, line2Atom2Point));
        String label = StandardGenerator.getAnnotationLabel(bond);
        if (label != null) {
            this.addAnnotation(atom1, atom2, label);
        }
        return group;
    }

    private IRenderingElement generateCrossedDoubleBond(IAtom from, IAtom to) {
        Point2d atom1BackOffPoint = this.backOffPoint(from, to);
        Point2d atom2BackOffPoint = this.backOffPoint(to, from);
        Vector2d unit = VecmathUtil.newUnitVector(atom1BackOffPoint, atom2BackOffPoint);
        Vector2d perpendicular1 = VecmathUtil.newPerpendicularVector(unit);
        Vector2d perpendicular2 = VecmathUtil.negate(perpendicular1);
        double halfSeparation = this.separation / 2.0;
        Vector2d line1Atom1Point = VecmathUtil.sum(atom1BackOffPoint, VecmathUtil.scale(perpendicular1, halfSeparation));
        Vector2d line1Atom2Point = VecmathUtil.sum(atom2BackOffPoint, VecmathUtil.scale(perpendicular1, halfSeparation));
        Vector2d line2Atom1Point = VecmathUtil.sum(atom1BackOffPoint, VecmathUtil.scale(perpendicular2, halfSeparation));
        Vector2d line2Atom2Point = VecmathUtil.sum(atom2BackOffPoint, VecmathUtil.scale(perpendicular2, halfSeparation));
        ElementGroup group = new ElementGroup();
        group.add(this.newLineElement(line1Atom1Point, line2Atom2Point));
        group.add(this.newLineElement(line2Atom1Point, line1Atom2Point));
        return group;
    }

    private IRenderingElement generateTripleBond(IBond bond, IAtom atom1, IAtom atom2) {
        ElementGroup group = new ElementGroup();
        Point2d p1 = this.backOffPoint(atom1, atom2);
        Point2d p2 = this.backOffPoint(atom2, atom1);
        Vector2d perp = VecmathUtil.newPerpendicularVector(VecmathUtil.newUnitVector(p1, p2));
        perp.scale(this.separation);
        group.add(new LineElement(p1.x, p1.y, p2.x, p2.y, this.stroke, this.foreground));
        group.add(new LineElement(p1.x + perp.x, p1.y + perp.y, p2.x + perp.x, p2.y + perp.y, this.stroke, this.foreground));
        group.add(new LineElement(p1.x - perp.x, p1.y - perp.y, p2.x - perp.x, p2.y - perp.y, this.stroke, this.foreground));
        String label = StandardGenerator.getAnnotationLabel(bond);
        if (label != null) {
            this.addAnnotation(atom1, atom2, label);
        }
        return group;
    }

    private IRenderingElement generateAttachPoint(IAtom atom, IBond bond) {
        Vector2d controlPoint2;
        Vector2d controlPoint1;
        Vector2d endPoint;
        double dist;
        int i;
        Point2d mid = atom.getPoint2d();
        Vector2d bndVec = VecmathUtil.newUnitVector(atom, bond);
        Vector2d bndXVec = VecmathUtil.newPerpendicularVector(bndVec);
        double length = atom.getPoint2d().distance(bond.getConnectedAtom(atom).getPoint2d());
        bndXVec.scale(length / 2.0);
        Vector2d beg = VecmathUtil.sum(atom.getPoint2d(), bndXVec);
        bndXVec.scale(-1.0);
        Vector2d end = VecmathUtil.sum(atom.getPoint2d(), bndXVec);
        int nCurves = (int)(2.0 * Math.ceil(length / this.waveSpacing));
        double step = length / (double)nCurves;
        bndXVec.normalize();
        Vector2d peak = VecmathUtil.scale(bndVec, step);
        Vector2d unit = VecmathUtil.newUnitVector(beg, end);
        ArrayList<PathElement> path = new ArrayList<PathElement>();
        int halfNCurves = nCurves / 2;
        path.add(new MoveTo(mid.x, mid.y));
        for (i = 1; i < halfNCurves; i += 2) {
            peak.negate();
            dist = (double)i * step;
            endPoint = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist)), peak);
            controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, (double)(i - 1) * step)), VecmathUtil.scale(peak, 0.5));
            controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, ((double)i - 0.5) * step)), peak);
            path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
            dist = (double)(i + 1) * step;
            endPoint = VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist));
            controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, ((double)i + 0.5) * step)), peak);
            controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist)), VecmathUtil.scale(peak, 0.5));
            path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
        }
        unit.negate();
        peak.negate();
        path.add(new MoveTo(mid.x, mid.y));
        for (i = 1; i < halfNCurves; i += 2) {
            peak.negate();
            dist = (double)i * step;
            endPoint = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist)), peak);
            controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, (double)(i - 1) * step)), VecmathUtil.scale(peak, 0.5));
            controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, ((double)i - 0.5) * step)), peak);
            path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
            dist = (double)(i + 1) * step;
            endPoint = VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist));
            controlPoint1 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, ((double)i + 0.5) * step)), peak);
            controlPoint2 = VecmathUtil.sum(VecmathUtil.sum(mid, VecmathUtil.scale(unit, dist)), VecmathUtil.scale(peak, 0.5));
            path.add(new CubicTo(controlPoint1.x, controlPoint1.y, controlPoint2.x, controlPoint2.y, endPoint.x, endPoint.y));
        }
        return new GeneralPath(path, this.foreground).outline(this.stroke);
    }

    private boolean isAttachPoint(IAtom atom) {
        return atom instanceof IPseudoAtom && ((IPseudoAtom)atom).getAttachPointNum() > 0;
    }

    private void addAnnotation(IAtom atom1, IAtom atom2, String label) {
        Vector2d perpendicular = VecmathUtil.newPerpendicularVector(VecmathUtil.newUnitVector(atom1.getPoint2d(), atom2.getPoint2d()));
        this.addAnnotation(atom1, atom2, label, perpendicular);
    }

    private void addAnnotation(IAtom atom1, IAtom atom2, String label, Vector2d perpendicular) {
        Point2d midPoint = VecmathUtil.midpoint(atom1.getPoint2d(), atom2.getPoint2d());
        TextOutline outline = StandardGenerator.generateAnnotation(midPoint, label, perpendicular, this.annotationDistance, this.annotationScale, this.font, null);
        this.annotations.add(MarkedElement.markup(GeneralPath.shapeOf(outline.getOutline(), this.annotationColor), "annotation"));
    }

    IRenderingElement generateDashedBond(IAtom from, IAtom to) {
        Point2d fromPoint = from.getPoint2d();
        Point2d toPoint = to.getPoint2d();
        Vector2d unit = VecmathUtil.newUnitVector(fromPoint, toPoint);
        int nDashes = (Integer)this.parameters.get(StandardGenerator.DashSection.class);
        double step = fromPoint.distance(toPoint) / (double)(3 * nDashes - 2);
        double start = this.hasDisplayedSymbol(from) ? fromPoint.distance(this.backOffPoint(from, to)) : Double.NEGATIVE_INFINITY;
        double end = this.hasDisplayedSymbol(to) ? fromPoint.distance(this.backOffPoint(to, from)) : Double.POSITIVE_INFINITY;
        ElementGroup group = new ElementGroup();
        double distance = 0.0;
        for (int i = 0; i < nDashes; ++i) {
            if (distance > start && distance + step < end) {
                group.add(this.newLineElement(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, distance)), VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, distance + step))));
            } else if (distance + step > start && distance + step < end) {
                group.add(this.newLineElement(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, start)), VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, distance + step))));
            } else if (distance > start && distance < end) {
                group.add(this.newLineElement(VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, distance)), VecmathUtil.sum(fromPoint, VecmathUtil.scale(unit, end))));
            }
            distance += step;
            distance += step;
            distance += step;
        }
        return group;
    }

    IRenderingElement newLineElement(Tuple2d a, Tuple2d b) {
        return new LineElement(a.x, a.y, b.x, b.y, this.stroke, this.foreground);
    }

    Point2d backOffPoint(IAtom from, IAtom to) {
        return StandardBondGenerator.backOffPointOf(this.symbols[this.atomIndexMap.get(from)], from.getPoint2d(), to.getPoint2d(), this.backOff);
    }

    boolean hasDisplayedSymbol(IAtom atom) {
        return this.symbols[this.atomIndexMap.get(atom)] != null;
    }

    static Point2d backOffPointOf(AtomSymbol symbol, Point2d fromPoint, Point2d toPoint, double backOff) {
        if (symbol == null) {
            return fromPoint;
        }
        Point2d intersect = VecmathUtil.toVecmathPoint(symbol.getConvexHull().intersect(VecmathUtil.toAwtPoint(fromPoint), VecmathUtil.toAwtPoint(toPoint)));
        if (intersect == null) {
            return fromPoint;
        }
        Vector2d unit = VecmathUtil.newUnitVector(fromPoint, toPoint);
        return new Point2d(VecmathUtil.sum(intersect, VecmathUtil.scale(unit, backOff)));
    }

    static int winding(IBond bond1, IBond bond2) {
        IAtom atom1 = bond1.getAtom(0);
        IAtom atom2 = bond1.getAtom(1);
        if (bond2.contains(atom1)) {
            return StandardBondGenerator.winding(atom2.getPoint2d(), atom1.getPoint2d(), bond2.getConnectedAtom(atom1).getPoint2d());
        }
        if (bond2.contains(atom2)) {
            return StandardBondGenerator.winding(atom1.getPoint2d(), atom2.getPoint2d(), bond2.getConnectedAtom(atom2).getPoint2d());
        }
        throw new IllegalArgumentException("Bonds do not share any atoms");
    }

    static Map<IBond, IAtomContainer> ringPreferenceMap(IAtomContainer container) {
        IRingSet relevantRings = Cycles.relevant(container).toRingSet();
        List<IAtomContainer> rings = AtomContainerSetManipulator.getAllAtomContainers(relevantRings);
        Collections.sort(rings, new RingBondOffsetComparator());
        HashMap<IBond, IAtomContainer> ringMap = new HashMap<IBond, IAtomContainer>();
        for (IAtomContainer ring : rings) {
            StandardBondGenerator.normalizeRingWinding(ring);
            for (IBond bond : ring.bonds()) {
                if (ringMap.containsKey(bond)) continue;
                ringMap.put(bond, ring);
            }
        }
        return Collections.unmodifiableMap(ringMap);
    }

    static void normalizeRingWinding(IAtomContainer container) {
        int prev = container.getAtomCount() - 1;
        int curr = 0;
        int next = 1;
        int n = container.getAtomCount();
        int winding = 0;
        while (curr < n) {
            winding += StandardBondGenerator.winding(container.getAtom(prev).getPoint2d(), container.getAtom(curr).getPoint2d(), container.getAtom(next % n).getPoint2d());
            prev = curr;
            curr = next++;
        }
        if (winding < 0) {
            IAtom[] atoms = new IAtom[n];
            for (int i = 0; i < n; ++i) {
                atoms[n - i - 1] = container.getAtom(i);
            }
            container.setAtoms(atoms);
        }
    }

    static int winding(Point2d a, Point2d b, Point2d c) {
        return (int)Math.signum((b.x - a.x) * (c.y - a.y) - (b.y - a.y) * (c.x - a.x));
    }

    static final class RingBondOffsetComparator
    implements Comparator<IAtomContainer> {
        private static final int[] PREFERENCE_INDEX = new int[8];

        RingBondOffsetComparator() {
        }

        @Override
        public int compare(IAtomContainer containerA, IAtomContainer containerB) {
            int sizeCmp = Ints.compare(RingBondOffsetComparator.sizePreference(containerA.getAtomCount()), RingBondOffsetComparator.sizePreference(containerB.getAtomCount()));
            if (sizeCmp != 0) {
                return sizeCmp;
            }
            int piBondCmp = Ints.compare(RingBondOffsetComparator.nDoubleBonds(containerA), RingBondOffsetComparator.nDoubleBonds(containerB));
            if (piBondCmp != 0) {
                return -piBondCmp;
            }
            int[] freqA = RingBondOffsetComparator.countLightElements(containerA);
            int[] freqB = RingBondOffsetComparator.countLightElements(containerB);
            for (Elements element : Arrays.asList(Elements.Carbon, Elements.Nitrogen, Elements.Oxygen, Elements.Sulfur, Elements.Phosphorus)) {
                int elemCmp = Ints.compare(freqA[element.number()], freqB[element.number()]);
                if (elemCmp == 0) continue;
                return -elemCmp;
            }
            return 0;
        }

        static int sizePreference(int size) {
            if (size < 3) {
                throw new IllegalArgumentException("a ring must have at least 3 atoms");
            }
            if (size > 7) {
                return size;
            }
            return PREFERENCE_INDEX[size];
        }

        static int nDoubleBonds(IAtomContainer container) {
            int count = 0;
            for (IBond bond : container.bonds()) {
                if (!IBond.Order.DOUBLE.equals((Object)bond.getOrder())) continue;
                ++count;
            }
            return count;
        }

        static int[] countLightElements(IAtomContainer container) {
            int[] freq = new int[19];
            for (IAtom atom : container.atoms()) {
                if (atom.getAtomicNumber() < 0 || atom.getAtomicNumber() >= 19) continue;
                int n = atom.getAtomicNumber();
                freq[n] = freq[n] + 1;
            }
            return freq;
        }

        static {
            int preference = 0;
            for (int size : new int[]{6, 5, 7, 4, 3}) {
                RingBondOffsetComparator.PREFERENCE_INDEX[size] = preference++;
            }
        }
    }
}

