#' @name exportUsers
#' @aliases exportUsers.redcapApiConnection
#' @aliases exportUsers.redcapDbConnection
#' @export exportUsers
#' @export exportUsers.redcapApiConnection
#' @export exportUsers.redcapDbConnection
#' @importFrom httr POST
#' 
#' @title Export the Users for a Project
#' @description Retrieve a data frame giving the users, expiration dates, 
#' and data access privileges for each user.
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param date Logical. Indicates if the expiration date is converted to a
#'   \code{POSIXct} object.
#' @param label Logical. Indicates if the data export and form access rights are
#'   converted to factor objects.
#' @param ... Arguments to be passed to other methods.
#' 
#' @details
#' From the REDCap API Documentation:
#' 
#' Data Export: 0=no access, 2=De-Identified, 1=Full Data Set\cr
#' Form Rights: 0=no access, 2=read only, 1=view records/responses and 
#' edit records (survey responses are read-only), 3 = edit survey responses
#' 
#' (NOTE: At this time, only a limited amount of rights-related info will be 
#' exported (expiration, data access group ID, data export rights, and 
#' form-level rights). However, more info about a user's rights will 
#' eventually be added to the Export Users API functionality in future 
#' versions of REDCap.)
#' 
#' For some reason I have yet to identify, some User Tables do not 
#' export correctly. In some situations, the fields are all shifted one 
#' column to the left and the form names are not always exported. 
#' This seems to be more common in projects still in Development mode. 
#' I have seen one instance of a project in Production where one user had 
#' one more column given than any other user.  If you notice this behavior, 
#' please report it to me as it may help me narrow down the source of the problem
#' 
#' @author Benjamin Nutter
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterbAPI/redcap/wiki/REDCap-API-Parameters}
#' 
#' @examples
#' \dontrun{
#' > #*** Note: I cannot provide working examples without
#' > #*** compromising security.  Instead, I will try to 
#' > #*** offer up sample code with the matching results
#' > 
#' > 
#' > #*** Create the connection object
#' > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#' > 
#' > exportUsers(rcon)
#' > #*** Note: I cannot provide working examples without
#' > #*** compromising security.  Instead, I will try to 
#' > #*** offer up sample code with the matching results
#' > 
#' > 
#' > #*** Create the connection object
#' > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#' > 
#' > exportUsers(rcon)
#' username            email firstname lastname expiration data_access_group
#' 1    user1 user1@@domain.org     Name1 Surname1       <NA>                NA
#' 2    user2             <NA>      <NA>     <NA>       <NA>                NA
#' 3    user3 user3@@domain.org     Name3 Surname3       <NA>                NA
#' 4    user4             <NA>      <NA>     <NA>       <NA>                NA
#' 5    user5 user5@@domain.org     Name5 Surname5       <NA>                NA
#' 6    user6             <NA>      <NA>     <NA>       <NA>                NA
#' 7    user7 user6@@domain.org     Name7 Surname7       <NA>                NA
#' data_export                 patient_characteristics
#' 1 Full data set view records/responses and edit records
#' 2 De-identified view records/responses and edit records
#' 3 De-identified view records/responses and edit records
#' 4 De-identified view records/responses and edit records
#' 5 Full data set view records/responses and edit records
#' 6 De-identified view records/responses and edit records
#' 7 Full data set view records/responses and edit records
#' dxa_scan_summary
#' 1 view records/responses and edit records
#' 2 view records/responses and edit records
#' 3 view records/responses and edit records
#' 4 view records/responses and edit records
#' 5 view records/responses and edit records
#' 6 view records/responses and edit records
#' 7 view records/responses and edit records
#' }
#' 




exportUsers <- function(rcon, ...) UseMethod("exportUsers")

#' @rdname exportUsers

exportUsers.redcapDbConnection <- function(rcon, date=TRUE, label=TRUE, ...){
  message("Please accept my apologies.  The exportUsers method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportUsers

exportUsers.redcapApiConnection <- function(rcon, date=TRUE, label=TRUE, ...){
  #* parameters for the Users File Export
  .params <- list(token=rcon$token, content='user', format='csv', returnFormat='csv')
  
  #* Export Users file and convert to data frame
  x <- httr::POST(url=rcon$url, body=.params, config=rcon$config)
  if (x$status_code != "200") stop(paste0(x$status_code, ": ", as.character(x)))
  x <- read.csv(textConnection(as.character(x)), stringsAsFactors=FALSE, na.strings="")
  
  #* convert expiration date to POSIXct class
  if (date) x$expiration <- as.POSIXct(x$expiration, format="%Y-%m-%d")
  
  #* convert data export and form editing privileges to factors
  if (label){
    x$data_export <- factor(x$data_export, c(0, 2, 1), c("No access", "De-identified", "Full data set"))
    
    x[, which(names(x) == "data_export"):length(x)] <- 
        lapply(x[, which(names(x) == "data_export"):length(x)], 
            factor, c(0, 2, 1, 3),
            c("No access", "Read only", "view records/responses and edit records", "Edit survey responses"))
  }
  
  return(x)
}
