#' @name deleteRecords
#' @title Delete Records From a Project
#' 
#' @description Delete records from a project.  This is a destructive action that
#'   will result in deletion of any events and data associated with the arm.
#'   Data loss is non-reversible.  The user must have 'Delete Record' privileges
#'   in the database.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param records a vector of record names to be deleted.
#' @param arm \code{integerish}, the arm number of the arm in which the 
#'   record(s) should be deleted. (This can only be used if the project is 
#'   longitudinal with more than one arm.) NOTE: If the arm parameter is not 
#'   provided, the specified records will be deleted from all arms in which 
#'   they exist. Whereas, if arm is provided, they will only be deleted from 
#'   the specified arm.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param ... Additional arguments to pass to other methods.
#' 
#' @section REDCap API Documentation:
#' This method allows you to delete one or more records from a project in a 
#' single API request.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' The number of deleted records.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' @export

deleteRecords <- function(rcon, 
                          records, 
                          arm = NULL, ...){
  UseMethod("deleteRecords")
}

#' @rdname deleteRecords
#' @export

deleteRecords.redcapApiConnection <- function(rcon, 
                                              records, 
                                              arm = NULL, ...,
                                              error_handling = getOption("redcap_error_handling")){
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_atomic(x = records,
                           any.missing = FALSE,
                           min.len = 1,
                           add = coll)
  
  checkmate::assert_integerish(arm,
                               null.ok = TRUE,
                               add = coll)
  
  checkmate::reportAssertions(coll)
  
  if (!is.character(records)){
    records <- as.character(records)
  }
  
  body <- list(token = rcon$token,
               content = "record",
               action = "delete")
  
  if(!is.null(arm)) body$arm <- arm
  
  body <- c(body,
            vectorToApiBodyList(vector = records,
                                parameter_name = "records"))
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200){
    return(redcap_error(x, error_handling))
  } 
  
  as.character(x)
}