#'Convert to an HTML document
#'
#'Format for converting from R Markdown to an HTML document.
#'
#'@param toc \code{TRUE} to include a table of contents in the output
#'@param toc_depth Depth of headers to include in table of contents
#'@param toc_float \code{TRUE} to float the table of contents to the left of the
#'  main document content. Rather than \code{TRUE} you may also pass a list of
#'  options that control the behavior of the floating table of contents. See the
#'  \emph{Floating Table of Contents} section below for details.
#'@param number_sections \code{TRUE} to number section headings
#'@param fig_width Default width (in inches) for figures
#'@param fig_height Default width (in inches) for figures
#'@param fig_retina Scaling to perform for retina displays (defaults to 2 when
#'  \code{fig_caption} is \code{FALSE}, which currently works for all widely
#'  used retina displays). Set to \code{NULL} to prevent retina scaling. Note
#'  that this will always be \code{NULL} when \code{keep_md} is specified (this
#'  is because \code{fig_retina} relies on outputting HTML directly into the
#'  markdown document).
#'@param fig_caption \code{TRUE} to render figures with captions
#'@param dev Graphics device to use for figure output (defaults to png)
#'@param code_folding Enable document readers to toggle the display of R code
#'  chunks. Defaults to \code{"none"} which displays all code chunks (assuming
#'  they were knit with \code{echo = TRUE}). Specify \code{"hide"} to hide all R
#'  code chunks by default (users can show hidden code chunks either
#'  individually or document-wide). Specify \code{"show"} to show all R code
#'  chunks by default.
#'@param smart Produce typographically correct output, converting straight
#'  quotes to curly quotes, --- to em-dashes, -- to en-dashes, and ... to
#'  ellipses.
#'@param self_contained Produce a standalone HTML file with no external
#'  dependencies, using data: URIs to incorporate the contents of linked
#'  scripts, stylesheets, images, and videos. Note that even for self contained
#'  documents MathJax is still loaded externally (this is necessary because of
#'  it's size).
#'@param theme Visual theme ("default", "cerulean", "journal", "flatly",
#'  "readable", "spacelab", "united", "cosmo", "lumen", "paper", "sandstone",
#'  "simplex", or "yeti"). Pass \code{NULL} for no theme (in this case you can
#'  use the \code{css} parameter to add your own styles).
#'@param highlight Syntax highlighting style. Supported styles include
#'  "default", "tango", "pygments", "kate", "monochrome", "espresso", "zenburn",
#'  "haddock", and "textmate". Pass \code{NULL} to prevent syntax highlighting.
#'@param mathjax Include mathjax. The "default" option uses an https URL from
#'  the official MathJax CDN. The "local" option uses a local version of MathJax
#'  (which is copied into the output directory). You can pass an alternate URL
#'  or pass \code{NULL} to exclude MathJax entirely.
#'@param template Pandoc template to use for rendering. Pass "default" to use
#'  the rmarkdown package default template; pass \code{NULL} to use pandoc's
#'  built-in template; pass a path to use a custom template that you've created.
#'  Note that if you don't use the "default" template then some features of
#'  \code{html_document} won't be available (see the Templates section below for
#'  more details).
#'@param css One or more css files to include
#'@param includes Named list of additional content to include within the
#'  document (typically created using the \code{\link{includes}} function).
#'@param keep_md Keep the markdown file generated by knitting.
#'@param lib_dir Directory to copy dependent HTML libraries (e.g. jquery,
#'  bootstrap, etc.) into. By default this will be the name of the document with
#'  \code{_files} appended to it.
#'@param md_extensions Markdown extensions to be added or removed from the
#'  default definition or R Markdown. See the \code{\link{rmarkdown_format}} for
#'  additional details.
#'@param pandoc_args Additional command line options to pass to pandoc
#'@param extra_dependencies,... Additional function arguments to pass to the
#'  base R Markdown HTML output formatter \code{\link{html_document_base}}
#'
#'@return R Markdown output format to pass to \code{\link{render}}
#'
#'@details
#'
#'See the \href{http://rmarkdown.rstudio.com/html_document_format.html}{online
#'documentation} for additional details on using the \code{html_document}
#'format.
#'
#'R Markdown documents can have optional metadata that is used to generate a
#'document header that includes the title, author, and date. For more details
#'see the documentation on R Markdown \link[=rmd_metadata]{metadata}.
#'
#'R Markdown documents also support citations. You can find more information on
#'the markdown syntax for citations in the
#'\href{http://rmarkdown.rstudio.com/authoring_bibliographies_and_citations.html}{Bibliographies
#'and Citations} article in the online documentation.
#'
#'
#'@section Navigation Bars:
#'
#'  If you have a set of html documents which you'd like to provide a common
#'  global navigation bar for, you can include a "_navbar.html" file within the
#'  same directory as your html document and it will automatically be included
#'  at the top of the document. For a simple example of including a navigation
#'  bar see
#'  .\href{https://github.com/rstudio/rmarkdown-website/blob/master/_navbar.html}{https://github.com/rstudio/rmarkdown-website/blob/master/_navbar.html}
#'   For additional documentation on creating Bootstrap navigation bars see
#'  \href{http://getbootstrap.com/components/#navbar}{http://getbootstrap.com/components/#navbar}.
#'
#'
#'@section Floating Table of Contents:
#'
#'  You may specify a list of options for the \code{toc_float} parameter which
#'  control the behavior of the floating table of contents. Options include:
#'
#'  \itemize{ \item{\code{collapsed} (defaults to \code{TRUE}) controls whether
#'  the table of contents appers with only the top-level (H2) headers. When
#'  collapsed the table of contents is automatically expanded inline when
#'  necessary.} \item{\code{smooth_scroll} (defaults to \code{TRUE}) controls
#'  whether page scrolls are animated when table of contents items are navigated
#'  to via mouse clicks.} }
#'
#'@section Tabbed Sections:
#'
#'  You can organize content using tabs by applying the \code{.tabset} class
#'  attribute to headers within a document. This will cause all sub-headers of
#'  the header with the \code{.tabset} attribute to appear within tabs rather
#'  than as standalone sections. For example:
#'
#'  \preformatted{ ## Quarterly Results {.tabset}
#'
#'  ### By Product
#'
#'  ### By Region }
#'
#'  You can also specify two additional attributes to control the appearance and
#'  behavior of the tabs. The \code{.tabset-fade} attributes causes the tabs to
#'  fade in and out when switching. The \code{.tabset-pills} attribute causes
#'  the visual appearance of the tabs to be "pill" rather than traditional tabs.
#'  For example:
#'
#'  \preformatted{ ## Quarterly Results {.tabset .tabset-fade .tabset-pills} }
#'
#'@section Templates:
#'
#'  You can provide a custom HTML template to be used for rendering. The syntax
#'  for templates is described in the
#'  \href{http://pandoc.org/README.html}{pandoc documentation}. You can also use
#'  the basic pandoc template by passing \code{template = NULL}.
#'
#'  Note however that if you choose not to use the "default" HTML template then
#'  several aspects of HTML document rendering will behave differently:
#'
#'  \itemize{ \item{The \code{theme} parameter does not work (you can still
#'  provide styles using the \code{css} parameter). } \item{For the
#'  \code{highlight} parameter, the default highlighting style will resolve to
#'  "pygments" and the "textmate" highlighting style is not available }
#'  \item{The \code{toc_float} parameter will not work. } \item{The
#'  \code{code_folding} parameter will not work. } \item{Tabbed sections (as
#'  described above) will not work.} \item{Navigation bars (as described above)
#'  will not work. }\item{MathJax will not work if \code{self_contained} is
#'  \code{TRUE} (these two options can't be used together in normal pandoc
#'  templates). } }
#'
#'  Due to the above restrictions, you might consider using the \code{includes}
#'  parameter as an alternative to providing a fully custom template.
#'
#' @examples
#' \dontrun{
#'
#' library(rmarkdown)
#'
#' render("input.Rmd", html_document())
#'
#' render("input.Rmd", html_document(toc = TRUE))
#' }
#'
#'@export
html_document <- function(toc = FALSE,
                          toc_depth = 3,
                          toc_float = FALSE,
                          number_sections = FALSE,
                          fig_width = 7,
                          fig_height = 5,
                          fig_retina = if (!fig_caption) 2,
                          fig_caption = FALSE,
                          dev = 'png',
                          code_folding = c("none", "show", "hide"),
                          smart = TRUE,
                          self_contained = TRUE,
                          theme = "default",
                          highlight = "default",
                          mathjax = "default",
                          template = "default",
                          extra_dependencies = NULL,
                          css = NULL,
                          includes = NULL,
                          keep_md = FALSE,
                          lib_dir = NULL,
                          md_extensions = NULL,
                          pandoc_args = NULL,
                          ...) {

  # build pandoc args
  args <- c("--standalone")

  # use section divs
  args <- c(args, "--section-divs")

  # table of contents
  args <- c(args, pandoc_toc_args(toc, toc_depth))

  # toc_float
  if (toc && !identical(toc_float, FALSE)) {

    # must have a theme
    if (is.null(theme))
      stop("You must use a theme when specifying the 'toc_float' option")

    # resolve options
    toc_float_options <- list(collapsed = TRUE,
                              smooth_scroll = TRUE)
    if (is.list(toc_float)) {
      toc_float_options <- merge_lists(toc_float_options, toc_float)
      toc_float <- TRUE
    } else if (!isTRUE(toc_float)) {
      stop("toc_float must be a logical or a list with options")
    }

    # dependencies
    extra_dependencies <- append(extra_dependencies,
                                 list(html_dependency_jquery(),
                                      html_dependency_jqueryui(),
                                      html_dependency_tocify()))

    # flag for template
    args <- c(args, pandoc_variable_arg("toc_float", "1"))

    # selectors
    selectors <- paste0("h", seq(1, toc_depth), collapse = ",")
    args <- c(args, pandoc_variable_arg("toc_selectors", selectors))

    # options
    if (toc_float_options$collapsed)
      args <- c(args, pandoc_variable_arg("toc_collapsed", "1"))
    if (toc_float_options$smooth_scroll)
      args <- c(args, pandoc_variable_arg("toc_smooth_scroll", "1"))
  }

  # template path and assets
  if (identical(template, "default"))
    args <- c(args, "--template",
              pandoc_path_arg(rmarkdown_system_file("rmd/h/default.html")))
  else if (!is.null(template))
    args <- c(args, "--template", pandoc_path_arg(template))

  # numbered sections
  if (number_sections)
    args <- c(args, "--number-sections")

  # additional css
  for (css_file in css)
    args <- c(args, "--css", pandoc_path_arg(css_file))

  # validate code_folding
  code_folding <- match.arg(code_folding)

  # pre-processor for arguments that may depend on the name of the
  # the input file (e.g. ones that need to copy supporting files)
  pre_processor <- function(metadata, input_file, runtime, knit_meta, files_dir,
                            output_dir) {

    # use files_dir as lib_dir if not explicitly specified
    if (is.null(lib_dir))
      lib_dir <- files_dir

    # extra args
    args <- c()

    # highlight
    args <- c(args, pandoc_html_highlight_args(highlight,
                                               template,
                                               self_contained,
                                               lib_dir,
                                               output_dir))

    # bootstrap navigation (requires theme)
    if (!is.null(theme)) {

      # js for for code folding and tabsets
      args <- c(args, pandoc_html_navigation_args(self_contained,
                                                  lib_dir,
                                                  output_dir))

      # add navbar to includes if necessary
      navbar <- file.path(normalize_path(dirname(input_file)), "_navbar.html")
      if (file.exists(navbar)) {
        if (is.null(includes))
          includes <- list()
        # include the navbar html
        includes$before_body <- c(navbar, includes$before_body)
        # flag indicating we need extra navbar css and js
        args <- c(args, pandoc_variable_arg("navbar", "1"))
        # variables controlling padding from navbar
        args <- c(args, pandoc_body_padding_variable_args(theme))
      }
    }

    # code_folding
    if (code_folding %in% c("show", "hide")) {
      # must have a theme
      if (is.null(theme))
        stop("You must use a theme when specifying the 'code_folding' option")
      args <- c(args, pandoc_variable_arg("code_folding", code_folding))
    }

    # content includes (we do this here so that user include-in-header content
    # goes after dependency generated content). make the paths absolute if
    # making a Shiny document so we can resolve them even if rendering
    # elsewhere.
    args <- c(args, includes_to_pandoc_args(includes,
                      filter = if (identical(runtime, "shiny"))
                        function(x) normalize_path(x, mustWork = FALSE)
                      else
                        identity))

    # return additional args
    args
  }

  # return format
  output_format(
    knitr = knitr_options_html(fig_width, fig_height, fig_retina, keep_md, dev),
    pandoc = pandoc_options(to = "html",
                            from = from_rmarkdown(fig_caption, md_extensions),
                            args = args),
    keep_md = keep_md,
    clean_supporting = self_contained,
    pre_processor = pre_processor,
    base_format = html_document_base(smart = smart, theme = theme,
                                     self_contained = self_contained,
                                     lib_dir = lib_dir, mathjax = mathjax,
                                     template = template,
                                     pandoc_args = pandoc_args,
                                     extra_dependencies = extra_dependencies,
                                     ...)
  )
}


#' Knitr options for an HTML output format
#'
#' Define knitr options for an R Markdown output format that creates
#' HTML output.
#'
#' @inheritParams html_document
#'
#' @return An list that can be passed as the \code{knitr} argument of the
#'   \code{\link{output_format}} function.
#'
#' @seealso \link{knitr_options}, \link{output_format}
#'
#' @export
knitr_options_html <- function(fig_width, fig_height, fig_retina, keep_md, dev = 'png') {

  opts_chunk <- list(dev = dev,
                     dpi = 96,
                     fig.width = fig_width,
                     fig.height = fig_height,
                     fig.retina = fig_retina)

  if (keep_md)
    opts_chunk$fig.retina <- NULL

  knitr_options(opts_chunk = opts_chunk)
}

themes <- function() {
  c("default",
    "cerulean",
    "journal",
    "flatly",
    "readable",
    "spacelab",
    "united",
    "cosmo",
    "lumen",
    "paper",
    "sandstone",
    "simplex",
    "yeti")
}

html_highlighters <- function() {
  c(highlighters(), "textmate")
}

default_mathjax <- function() {
  paste("https://cdn.mathjax.org/mathjax/latest/",
        mathjax_config(), sep="")
}

mathjax_config <- function() {
  "MathJax.js?config=TeX-AMS-MML_HTMLorMML"
}

# variable which controls body offset (depends on height of navbar in theme)
pandoc_body_padding_variable_args <- function(theme) {

  # height of navbar in bootstrap 3.3.5
  navbarHeights <- c("default" = 51,
                     "cerulean" = 51,
                     "journal" = 61 ,
                     "flatly" = 60,
                     "readable" = 66,
                     "spacelab" = 52,
                     "united" = 51,
                     "cosmo" = 51,
                     "lumen" = 54,
                     "paper" = 64,
                     "sandstone" = 61,
                     "simplex" = 41,
                     "yeti" = 45)

  # body padding is navbar height
  bodyPadding <- navbarHeights[[theme]]

  # header padding is bodyPadding + 5
  headerPadding <- bodyPadding + 5

  # return variables
  c(pandoc_variable_arg("body_padding", bodyPadding),
    pandoc_variable_arg("header_padding", headerPadding))
}




