% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hl2Test.R
\name{hl2_test}
\alias{hl2_test}
\title{Two-sample location tests based on two-sample Hodges-Lehmann estimator.}
\usage{
hl2_test(
  x,
  y,
  alternative = c("two.sided", "greater", "less"),
  delta = ifelse(var.test, 1, 0),
  method = c("asymptotic", "permutation", "randomization"),
  scale = c("S1", "S2"),
  n.rep = 10000,
  na.rm = FALSE,
  var.test = FALSE,
  wobble = FALSE,
  wobble.seed = NULL
)
}
\arguments{
\item{x}{a (non-empty) numeric vector of data values.}

\item{y}{a (non-empty) numeric vector of data values.}

\item{alternative}{a character string specifying the alternative hypothesis,
must be one of "two.sided" (default), "greater", or "less".}

\item{delta}{a numeric value indicating the true difference in the location or
scale parameter, depending on whether the test should be performed
for a difference in location or in scale. The default is
\code{delta = 0} for a location test and \code{delta = 1} for a scale
test. In case of \code{var.test = TRUE}, \code{delta} represents
the ratio of the squared scale parameters.}

\item{method}{a character string specifying how the p-value is computed with
possible values \code{"asymptotic"} for an asymptotic test
based on a normal approximation, \code{"permutation"} for a
permutation test, and \code{"randomization"} for a randomization
test. The permutation test uses all splits of the joint sample
into two samples of sizes \code{m} and \code{n}, while the
randomization test draws \code{n.rep} random splits with
replacement. The values \code{m} and \code{n} denote the
sample sizes.
If not specified explicitly, defaults to
\code{"permutation"}  if \code{m < 30}, \code{n < 30} and
 \code{n.rep >= choose(m + n, m)},
\code{"randomization"} if \code{m < 30}, \code{n < 30} and
 \code{n.rep < choose(m + n, m)}, and
 \code{"asymptotic"} if \code{m >= 30} and \code{n >= 30}.}

\item{scale}{a character string specifying the scale estimator used for standardization
of the test statistic; must be one of \code{"S1"} and \code{"S2"}.
The default is \code{"S1"}. Ignored if \code{method = "asymptotic"};
see details for the definition of the scale estimators.}

\item{n.rep}{an integer value specifying the number of random splits used to
calculate the randomization distribution if \code{method = "randomization"}.
This argument is ignored if \code{method = "permutation"} or \code{method = "asymptotic"}.
The default is \code{n.rep = 10000}.}

\item{na.rm}{a logical value indicating whether NA values in \code{x} and
\code{y} should be stripped before the computation proceeds. The
default is \code{na.rm = FALSE}.}

\item{var.test}{a logical value to specify if the samples should be compared
for a difference in scale. The default is \code{var.test = FALSE}.}

\item{wobble}{a logical value indicating whether the sample should be checked for
duplicated values that can cause the scale estimate to be zero.
If such values are present, uniform noise is added to the sample,
see \code{\link[robnptests]{wobble}}.
Only necessary for the permutation and randomization version of the test.
The default is \code{wobble = FALSE}.}

\item{wobble.seed}{an integer value used as a seed for the random number
generation in case of \code{wobble = TRUE} or when
\code{var.test = TRUE} with one of the vectors \code{x} and
\code{y} containing zeros. When no seed is specified, it
is chosen randomly and printed in a message. The argument
is ignored if \code{var.test = FALSE} and/or \code{wobble = FALSE}.}
}
\value{
A named list with class "\code{htest}" containing the following components:
\item{statistic}{the value of the test statistic.}
\item{p.value}{the p-value for the test.}
\item{estimate}{the estimated location difference between \code{x} and \code{y}
                (if \code{var.test = FALSE}) or of \code{log(x^2)} and
                \code{log(y^2)} (if \code{var.test = TRUE}) based on the
                two-sample Hodges-Lehmann estimator.}
\item{null.value}{the specified hypothesized value of the mean difference/squared
                  scale ratio.}
\item{alternative}{a character string describing the alternative hypothesis.}
\item{method}{a character string indicating how the p-value was computed.}
\item{data.name}{a character string giving the names of the data.}
}
\description{
\code{hl2_test} performs a two-sample location test based on the two-sample
Hodges-Lehmann estimator for shift.
}
\details{
The test statistic for this test is based on the two-sample Hodges-Lehmann
estimator of \code{x} and \code{y}, see
\code{\link[robnptests]{hodges_lehmann_2sample}}. Three versions of the test
are implemented: randomization, permutation, and asymptotic.

The test statistic for the permutation and randomization version of the test
is standardized using a robust scale estimator, see
\insertCite{FriDeh11robu}{robnptests}.

With \code{scale = "S1"}, the scale is estimated by

\deqn{S = med(|x_i - x_j|: 1 \le i < j \le m, |y_i - y_j|, 1 \le i < j \le n),}

whereas \code{scale = "S2"} uses

\deqn{S = med(|z_i - z_j|: 1 \le i < j \le m + n).}

Here, \eqn{z = (z_1, ..., z_{m + n}) = (x_1 - med(x), ..., x_m - med(x), y_1 - med(y), ..., y_n - med(y))}
is the median-corrected sample.

The randomization distribution is based on randomly drawn splits with
replacement. The function \code{\link[statmod]{permp}} \insertCite{PhiSmy10perm}{robnptests}
is used to calculate the p-value. For the asymptotic test, a transformed version
of the HL2-estimator, which asymptotically follows a normal distribution, is
used. For more details on the asymptotic test, see \insertCite{FriDeh11robu;textual}{robnptests}.

For \code{var.test = TRUE}, the test compares the two samples for a difference
in scale. This is achieved by log-transforming the original squared observations,
i.e. \code{x} is replaced by \code{log(x^2)} and \code{y} by \code{log(y^2)}.
A potential scale difference then appears as a location difference between
the transformed samples, see \insertCite{Fri12onli;textual}{robnptests}.
The sample should not contain zeros to prevent problems with the necessary
log-transformation. If it contains zeros, uniform noise is added to all
variables in order to remove zeros and a message is printed.

If the sample has been modified (either because of zeros if \code{var.test = TRUE}
or \code{wobble = TRUE}), the modified samples can be retrieved using

\code{set.seed(wobble.seed); wobble(x, y)}.

Both samples need to contain at least 5 non-missing values.
}
\examples{
# Generate random samples
set.seed(108)
x <- rnorm(20); y <- rnorm(20)

# Asymptotic HL2 test
hl2_test(x, y, method = "asymptotic", scale = "S1")

\dontrun{
# HL22 test using randomization principle by drawing 1000 random permutations
# with replacement

hl2_test(x, y, method = "randomization", n.rep = 1000, scale = "S2")
}

}
\references{
\insertRef{PhiSmy10perm}{robnptests}

\insertRef{FriDeh11robu}{robnptests}

\insertRef{Fri12onli}{robnptests}
}
