#plot slippy tiles

tile.loadimage <- function(x, y, zoom, type, cachedir=NULL) {
  if(x < 0) {
    #negative tiles from wrap situation
    x <- x+2^zoom
  }

  fname <- tile.cachename(x, y, zoom, type, cachedir)
  parts <- strsplit(fname, "\\.")[[1]]
  ext <- parts[length(parts)]
  tryCatch({
    if(ext == "jpg" || ext =="jpeg") {
      return(jpeg::readJPEG(fname))
    } else if(ext == "png") {
      return(png::readPNG(fname))
    } else {
      stop("Extension not recognized: ", ext)
    }
  }, error=function(err) {
    message("Error loading ", fname, ": ", err)
  })
  NULL
}

tile.applywrap <- function(tiles, zoom) {
  if(!all(min(tiles[,1]):max(tiles[,1]) %in% tiles[,1])) {
    #wrapping around the backside of the earth, make end tiles negative
    warning("Attempting to plot wrap around tiles (~lat=-180), things may get funky.")
    n <- -1
    while(length(tiles[,1][tiles[,1]==2^zoom+n]) > 0) {
      tiles[,1][tiles[,1]==2^zoom+n] <- (tiles[,1][tiles[,1]==2^zoom+n]) - 2^zoom
      n <- n-1
    }
  }
  tiles
}

tile.each <- function(tiles, zoom, type, epsg=4326, cachedir=NULL, plot=FALSE) {

  tiles <- tile.applywrap(tiles, zoom)
  outlist <- list()

  for(i in 1:nrow(tiles)) {
    x <- tiles[i,1]
    y <- tiles[i,2]
    box <- tile.bbox(x, y, zoom, epsg)
    image <- tile.loadimage(x, y, zoom, type, cachedir)

    if(!is.null(image)) {
      if(plot) {
        tile.plotarray(image, box)
      } else {
        outlist[[length(outlist)+1]] <- list(img=image, bbox=box)
      }
    }
  }

  if(!plot) {
    outlist
  }
}

tile.arbind <- function(...) {
  abind::abind(..., along=1)
}

tile.acbind <- function(...) {
  abind::abind(..., along=2)
}

tile.fuse <- function(tiles, zoom, type, epsg=4326, cachedir=NULL, plot=FALSE) {

  tiles <- tile.applywrap(tiles, zoom)

  tiles <- tiles[order(tiles$Var1, tiles$Var2),]
  xs <- unique(tiles[,1])
  ys <- unique(tiles[,2])

  `%do%` <- foreach::`%do%` #dopar actually decreases performance
  x <- NULL; y<-NULL; rm(x); rm(y) #CMD trick
  wholeimg <- foreach::foreach(x=xs, .combine=tile.acbind, .multicombine = TRUE) %do% {
                foreach::foreach(y=ys, .combine=tile.arbind, .multicombine = TRUE) %do% {
                  img <- tile.loadimage(x, y, zoom, type, cachedir)
                  if(is.null(img)) stop("Cannot fuse unloadable tile: ", x, ",", y)
                  img
                }
              }

  #calc bounding box of whole image

  nw <- tile.nw(min(xs), min(ys), zoom, epsg)
  se <- tile.nw(max(xs)+1, max(ys)+1, zoom, epsg)

  bbox <- matrix(c(nw[1], se[2], se[1], nw[2]), ncol=2,
                byrow=FALSE, dimnames=list(c("x", "y"), c("min", "max")))

  if(plot) {
    #plot image
    tile.plotarray(wholeimg, bbox)
  } else {
    #return image and bounds
    list(img=wholeimg, bbox=bbox)
  }
}


#' Get List of Valid Types
#'
#' @return A character vector of valid \code{type} parameters.
#'
#' @export
#'
#' @examples
#' osm.types()
#'
osm.types <- function() {
  c("hikebike","hillshade","hotstyle","lovinacycle",
     "lovinahike","mapquestosm","mapquestsat","opencycle",
     "openpiste","osm","osmgrayscale",
     "osmtransport","stamenbw","stamenwatercolor",
     "thunderforestlandscape","thunderforestoutdoors")
}

#' Plot Open Street Map Tiles
#'
#' Plot Open Street Map tiles using \code{rasterImage} and \code{sp::plot}.
#' Define your own tile sources by creating a tile url function in the
#' global environment, although most \href{http://wiki.openstreetmap.org/wiki/Tile_servers}{OSM listed}
#' servers are included. See \link{osm.types} for types options. By default tiles
#' are plotted in the Spherical Mercator projection
#' (\href{https://en.wikipedia.org/wiki/Web_Mercator}{epsg:3857}); pass \code{project=FALSE}
#' to keep lat/lon coordinates.
#'
#' @param bbox A bounding box as generated by \code{sp::bbox()} or \code{prettymapr::searchbbox()}
#' @param zoomin The amount by which to adjust the automatically calculated zoom (or
#' manually specified if the \code{zoom} parameter is passed). Use +1 to zoom in, or -1 to zoom out.
#' @param zoom Manually specify the zoom level (not reccomended; adjust \code{zoomin} or
#' \code{res} instead.
#' @param type A map type; one of that returned by \link{osm.types}. User defined types are possible
#' by defining \code{tile.url.TYPENAME <- function(xtile, ytile, zoom){}} and passing TYPENAME
#' as the \code{type} argument.
#' @param forcedownload \code{TRUE} if cached tiles should be re-downloaded. Useful if
#' some tiles are corrupted.
#' @param stoponlargerequest By default \code{osm.plot} will only load 32 tiles at a time. If
#' plotting at a higher resolution it may be necessary to pass \code{true} here.
#' @param fusetiles \code{TRUE} if tiles should be fused into a single image. This is the
#' default because white lines appear between tiles if it is set to \code{FALSE}. PDFs
#' appear not to have this problem, so when plotting large, high resolution PDFs it may be
#' faster (and more memory efficient) to use \code{fusetiles=FALSE}.
#' @param cachedir The directory in which tiles should be cached. Defaults to \code{getwd()/rosm.cache}.
#' @param res The resolution used to calculate scale.
#' @param project \code{TRUE} if tiles should be projected to a pseudo-mercator projection,
#' \code{FALSE} if lat/lon should be maintained. Becuase \code{sp::plot} adjusts the aspect
#' according to latitude for lat/lon coordinates, this makes little difference at high
#' zoom and may make plotting overlays more convenient. Defaults to \code{TRUE}.
#' @param ... Additional parameters to be passed on to the first call to \code{sp::plot}
#'
#' @export
#'
#' @examples
#' \donttest{
#' library(prettymapr)
#' ns <- makebbox(47.2, -59.7, 43.3, -66.4)
#' osm.plot(ns)
#' osm.plot(ns, type="mapquestsat")
#' prettymap(osm.plot(ns), scale.style="ticks", scale.tick.cex=0)
#'
#' tile.url.darkmatter <- function(xtile, ytile, zoom) {
#'   paste0(paste("http://a.basemaps.cartocdn.com/dark_all",
#'                             zoom, xtile, ytile, sep="/"), ".png")
#' }
#' osm.plot(ns, type="darkmatter")
#' }
osm.plot <- function(bbox, zoomin=0, zoom=NULL, type="osm", forcedownload=FALSE,
                     stoponlargerequest=TRUE, fusetiles=TRUE, cachedir=NULL, res=150,
                     project=TRUE, ...) {

  if(project) {
    epsg <- 3857
  } else {
    epsg <- 4326
  }

  bbox <- .projectbbox(bbox, epsg)

  coords <- sp::coordinates(t(bbox))
  spoints = sp::SpatialPoints(coords, proj4string = sp::CRS(paste0("+init=epsg:", epsg)))

  plotargs <- list(...)
  if(is.null(plotargs$xlim))
    xlim <- bbox[1,]
  if(is.null(plotargs$ylim))
    ylim <- bbox[2,]

  sp::plot(spoints, pch=".", xlim=xlim, ylim=ylim, ...)

  if(is.null(zoom)) {
    zoom <- tile.autozoom(res=res, epsg=epsg)
  }
  zoom <- zoom+zoomin
  maxzoom <- tile.maxzoom(type)
  zoom <- min(zoom, maxzoom)

  #global min zoom set to 1
  zoom <- max(1, zoom)
  message("Zoom: ", zoom)
  #adjust bbox to final plot extents
  bbox <- t(matrix(graphics::par('usr'), ncol=2, byrow=FALSE))

  tiles <- tiles.bybbox(bbox, zoom, epsg=epsg)
  if((nrow(tiles)>32) && stoponlargerequest) stop("More than 32 tiles to be loaded. ",
                                                  "Run with stoponlargerequest=FALSE or ",
                                                  "zoomin=-1, to continue")
  tile.download(tiles, zoom, type=type, forcedownload=forcedownload, cachedir=cachedir)

  if(fusetiles) {
    tile.fuse(tiles, zoom, type=type, epsg=epsg, cachedir=cachedir, plot=TRUE)
  } else {
    tile.each(tiles, zoom, type=type, epsg=epsg, cachedir=cachedir, plot=TRUE)
  }

  tile.attribute(type)
}
