% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ResidualCorr.R
\name{ResidualCorr}
\alias{ResidualCorr}
\title{Compute the residual correlation and its significance}
\usage{
ResidualCorr(
  exp,
  obs,
  ref,
  N.eff = NA,
  time_dim = "sdate",
  memb_dim = NULL,
  method = "pearson",
  alpha = NULL,
  handle.na = "return.na",
  ncores = NULL
)
}
\arguments{
\item{exp}{A named numerical array of the forecast with at least time 
dimension.}

\item{obs}{A named numerical array of the observations with at least time 
dimension. The dimensions must be the same as "exp" except 'memb_dim'.}

\item{ref}{A named numerical array of the reference forecast data with at 
least time dimension. The dimensions must be the same as "exp" except 
'memb_dim'.}

\item{N.eff}{Effective sample size to be used in the statistical significance
test. It can be NA (and it will be computed with the s2dv:::.Eno), a 
numeric (which is used for all cases), or an array with the same dimensions
as "obs" except "time_dim" (for a particular N.eff to be used for each case)
. The default value is NA.}

\item{time_dim}{A character string indicating the name of the time dimension.
The default value is 'year'.}

\item{memb_dim}{A character string indicating the name of the member dimension
to compute the ensemble mean of the forecast and reference forecast. If it 
is NULL, the ensemble mean should be provided directly to the function. The
default value is NULL.}

\item{method}{A character string indicating the correlation coefficient to be
computed ("pearson", "kendall", or "spearman"). The default value is 
"pearson".}

\item{alpha}{A numeric of the significance level to be used in the statistical
significance test. If it is a numeric, "sign" will be returned. If NULL, the
p-value will be returned instead. The default value is NULL.}

\item{handle.na}{A charcater string indicating how to handle missing values.
If "return.na", NAs will be returned for the cases that contain at least one
NA in "exp", "ref", or "obs". If "only.complete.triplets", only the time 
steps with no missing values in all "exp", "ref", and "obs" will be used. If
"na.fail", an error will arise if any of "exp", "ref", or "obs" contains any
NA. The default value is "return.na".}

\item{ncores}{An integer indicating the number of cores to use for parallel 
computation. The default value is NULL.}
}
\value{
A list with:
\item{$res.corr}{
 A numerical array of the residual correlation with the same dimensions as 
 the input arrays except "time_dim" (and "memb_dim" if provided).
}
\item{$sign}{
 A logical array indicating whether the residual correlation is statistically 
 significant or not with the same dimensions as the input arrays except "time_dim"
 (and "memb_dim" if provided). Returned only if "alpha" is a numeric.
}
\item{$p.val}{
 A numeric array of the p-values with the same dimensions as the input arrays
 except "time_dim" (and "memb_dim" if provided). Returned only if "alpha" is
 NULL.
}
}
\description{
The residual correlation assesses whether a forecast captures any of the 
observed variability that is not already captured by a reference forecast 
(Smith et al., 2019; https://doi.org/10.1038/s41612-019-0071-y.).
The procedure is as follows: the residuals of the forecasts and observations 
are computed by linearly regressing out the reference forecast's ensemble mean
from the forecasts' ensemble mean and observations, respectively. Then, the 
residual correlation is computed as the correlation between both residuals. 
Positive values of the residual correlation indicate that the forecast capture
more observed variability than the reference forecast, while negative values  
mean that the reference forecast capture more. The significance of the 
residual correlation is computed with a two-sided t-test
(Wilks, 2011; https://doi.org/10.1016/B978-0-12-385022-5.00008-7) using an 
effective degrees of freedom to account for the time series' autocorrelation 
(von Storch and Zwiers, 1999; https://doi.org/10.1017/CBO9780511612336).
}
\examples{
exp <- array(rnorm(1000), dim = c(lat = 3, lon = 2, member = 10, sdate = 50))
obs <- array(rnorm(1000), dim = c(lat = 3, lon = 2, sdate = 50))
ref <- array(rnorm(1000), dim = c(lat = 3, lon = 2, member = 5, sdate = 50))
res <- ResidualCorr(exp = exp, obs = obs, ref = ref, memb_dim = 'member')

}
