% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/samc.R
\name{samc}
\alias{samc}
\alias{samc,RasterLayer,RasterLayer,RasterLayer,logical,function,missing-method}
\alias{samc,RasterLayer,RasterLayer,missing,logical,function,missing-method}
\alias{samc,matrix,matrix,matrix,missing,function,missing-method}
\alias{samc,matrix,matrix,missing,missing,function,missing-method}
\alias{samc,missing,missing,missing,missing,missing,dgCMatrix-method}
\alias{samc,missing,missing,missing,missing,missing,matrix-method}
\title{Create an samc object}
\usage{
samc(resistance, absorption, fidelity, latlon, tr_fun, p_mat, ...)

\S4method{samc}{RasterLayer,RasterLayer,RasterLayer,logical,`function`,missing}(
  resistance,
  absorption,
  fidelity,
  latlon,
  tr_fun,
  override = FALSE,
  directions = 8
)

\S4method{samc}{RasterLayer,RasterLayer,missing,logical,`function`,missing}(resistance, absorption, latlon, tr_fun, override = FALSE, directions = 8)

\S4method{samc}{matrix,matrix,matrix,missing,`function`,missing}(
  resistance,
  absorption,
  fidelity,
  tr_fun,
  override = FALSE,
  directions = 8
)

\S4method{samc}{matrix,matrix,missing,missing,`function`,missing}(resistance, absorption, tr_fun, override = FALSE, directions = 8)

\S4method{samc}{missing,missing,missing,missing,missing,dgCMatrix}(p_mat, override = FALSE)

\S4method{samc}{missing,missing,missing,missing,missing,matrix}(p_mat, override = FALSE)
}
\arguments{
\item{resistance}{A \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}}}

\item{absorption}{A \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}}}

\item{fidelity}{A \code{\link[raster]{RasterLayer-class}} or \code{\link[base]{matrix}}}

\item{latlon}{Logical (\code{TRUE} or \code{FALSE}) indicating whether the rasters use latitude/longitude}

\item{tr_fun}{A function to calculate the transition values in the \code{\link[gdistance]{transition}} function}

\item{p_mat}{A base R \code{\link[base]{matrix}} object or Matrix package dgCMatrix sparse matrix}

\item{...}{Placeholder}

\item{override}{Optional flag to prevent accidentally running memory intensive functions. Defaults to \code{FALSE}}

\item{directions}{Optional param. Must be set to either 4 or 8 (default is 8)}
}
\value{
A spatial absorbing Markov chain object
}
\description{
Create an samc object that contains the absorbing Markov chain data
}
\details{
This function is used to create a \code{\link{samc-class}} object. There are
two options for creating this object.

\strong{Option 1: Raster and Matrix Inputs}

The \code{\link{samc-class}} object can be created from a combination of
resistance, absorption, and fidelity data. These different landscape data
inputs must be the same type (a matrix or RasterLayer), and have identical
properties, including dimensions, location of NA cells, and CRS (if using
RasterLayers). Some of the inputs are mandatory, whereas others are optional.

The \code{resistance} and \code{absorption} inputs are always mandatory, whereas the
\code{fidelity} input is optional. If the \code{fidelity} input is not provided, then it it
is assumed that there is no site fidelity (i.e., individuals will always move
to an adjacent cell each time step).

The \code{latlon} parameter is required if the landscape data inputs are RasterLayer
objects. The package does not attempt to determine this automatically, and it
does not assume a default. Users must set it to \code{TRUE} if they are using
latitude and longitude data.

The \code{tr_fun} parameter is mandatory. It used when calculating the values for
the transition matrix. Internally, this is passed to the \code{\link[gdistance]{transition}}
function in the gdistance package to create the transition matrix.

\strong{Option 2: P Matrix Input}

The \code{p_mat} parameter can be used to create a \code{\link{samc-class}} object
directly from a preconstructed P matrix. This matrix must be either a base R
matrix, or a sparse matrix (dgCMatrix format) from the Matrix package. It
must meet the requirement of a P matrix described in Fletcher et al. (2019).
This includes:
\itemize{
  \item The number of rows must equal the number of columns (a square matrix)
  \item The last row must contain all 0's, except the last element, which must be 1
  \item Each row must sum to 1
  \item All values must be in the range of 0-1
}

Additionally, the columns and rows of the P matrix may be named (e.g., using
dimnames(), rowname(), colnames(), etc). When specifying \code{origin} or \code{dest} inputs
to metrics, these names may be used instead of cell numbers. This has the
advantage of making the code for an analysis easier to read and interpret,
which may also help to eliminate unintentional mistakes. There are two
requirements for naming the rows/cols of a P matrix. First, since the P matrix
represents a pairwise matrix, the row and column names must be the same. Second,
there must be no duplicate names. The exception to these rules is the very last
column and the very last row of the P matrix. Since these are not part of the
pairwise transition matrix, they may have whatever names the user prefers.

\strong{Other Parameters}

The \code{override} parameter is optional. To prevent users from unintentionally
running memory intensive versions of functions that could make their systems
non-responsive or crash software, it is set to \code{FALSE} by default. For various
reasons, it can be set to \code{TRUE}. In particular, a user might do this if they
are using a very small landscape dataset, or perhaps for a moderately sized
dataset if they have access to a system with exceptionally large amounts of
RAM. Before setting this to \code{TRUE}, users should read the Performance vignette/
article to understand the expected memory requirements. They should also
consider starting with scaled down version of their data and then gradually
scaling back up while monitoring their memory usage as a means to gauge what
is reasonable for their system.

The \code{directions} parameter is optional. When constructing the P matrix from
matrix or raster data, the \code{samc()} function must decide how adjacent cells are
connected. This value can be set to either 4 or 8. When set to 4, nodes are
connected horizontally and vertically (similar to the directions of how a rook
moves in chess). When set to 8, nodes are connected diagonally in addition to
horizontally and vertically (queen movement in chess). When not specified,
the \code{samc()} function defaults to a value of 8. When using large datasets to
construct a P matrix, setting the directions to 4 may lead to significant
improvements in computation time and the amount of memory needed to perform
an analysis.

\strong{Additional Information}

Depending on the data used to construct the samc-class object, some metrics
may cause crashes. This is a result of the underlying P matrix having specific
properties that make some equations unsolvable. One known case is a P matrix
that represents a disconnected graph, which can lead to the \code{cond_passage()}
function crashing. In terms of raster/matrix inputs, a disconnected graph
occurs when one or more pixels/cells are unreachable from other pixels/cells
due to the presence of a full barrier made up of NA values. In a raster, these
may be obvious as islands, but can be as inconspicuous as a rogue isolated
pixel. There may be other currently unknown situations that lead to unsolvable
metrics.

Future work is planned towards identifying these issues during creation of the
samc-class object and handling them appropriately to prevent inadvertent
crashes.
}
\examples{
# "Load" the data. In this case we are using data built into the package.
# In practice, users will likely load raster data using the raster() function
# from the raster package.
res_data <- samc::ex_res_data
abs_data <- samc::ex_abs_data
occ_data <- samc::ex_occ_data


# Make sure our data meets the basic input requirements of the package using
# the check() function.
check(res_data, abs_data)
check(res_data, occ_data)


# Create a `samc-class` object with the resistance and absorption data using
# the samc() function. We use the recipricol of the arithmetic mean for
# calculating the transition matrix. Note, the input data here are matrices,
# not RasterLayers. If using RasterLayers, the latlon parameter must be set.
samc_obj <- samc(res_data, abs_data, tr_fun = function(x) 1/mean(x))


# Convert the occupancy data to probability of occurrence
occ_prob_data <- occ_data / sum(occ_data, na.rm = TRUE)


# Calculate short- and long-term metrics using the analytical functions
short_mort <- mortality(samc_obj, occ_prob_data, time = 50)
short_dist <- distribution(samc_obj, origin = 3, time = 50)
long_disp <- dispersal(samc_obj, occ_prob_data)
visit <- visitation(samc_obj, dest = 4)
surv <- survival(samc_obj)


# Use the map() function to turn vector results into RasterLayer objects.
short_mort_map <- map(samc_obj, short_mort)
short_dist_map <- map(samc_obj, short_dist)
long_disp_map <- map(samc_obj, long_disp)
visit_map <- map(samc_obj, visit)
surv_map <- map(samc_obj, surv)
}
