% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_Discrete.R
\name{tree_discrete}
\alias{tree_discrete}
\title{Phylogenetic uncertainty - Trait Evolution Discrete Characters}
\usage{
tree_discrete(data, phy, n.tree = 10, model, transform = "none",
  bounds = list(), n.cores = NULL, track = TRUE, ...)
}
\arguments{
\item{data}{Data vector for a single binary trait, with names matching tips in \code{phy}.}

\item{phy}{Phylogenies (class 'multiPhylo', see ?\code{ape}).}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file. If NULL, \code{n.tree} = 10}

\item{model}{The Mkn model to use (see Details).}

\item{transform}{The evolutionary model to transform the tree (see Details). Default is \code{none}.}

\item{bounds}{settings to constrain parameter estimates. See \code{\link[geiger]{fitDiscrete}}}

\item{n.cores}{number of cores to use. If 'NULL', number of cores is detected.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[geiger]{fitDiscrete}}}
}
\value{
The function \code{tree_discrete} returns a list with the following
components:

\code{call}: The function call

\code{data}: The original full data vector

\code{sensi.estimates}: Parameter estimates (transition rates q12 and q21), 
AICc and the optimised value of the phylogenetic transformation parameter (e.g. \code{lambda}) 
for each analysis with a different phylogenetic tree.

\code{N.tree}: Number of trees \code{n.tree} analysed

\code{stats}: Main statistics for model parameters, i.e. minimum, maximum, mean, median and sd-values

\code{optpar}: Transformation parameter used (e.g. \code{lambda}, \code{kappa} etc.)
}
\description{
Fits models for trait evolution of discrete (binary) characters, 
evaluating phylogenetic uncertainty.
}
\details{
This function fits different models of discrete character evolution using \code{\link[geiger]{fitDiscrete}}
to n trees, randomly picked in a multiPhylo file. Currenlty, only binary discrete traits are supported

Different character model from \code{fitDiscrete} can be used, including \code{ER} (equal-rates), 
\code{SYM} (symmetric), \code{ARD} (all-rates-different) and \code{meristic} (stepwise fashion). 

All transformations to the phylogenetic tree from \code{fitDiscrete} can be used, i.e. \code{none},
\code{EB}, \code{lambda}, \code{kappa} and\code{delta}.

See \code{\link[geiger]{fitDiscrete}} for more details on character models and tree transformations. 

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
#Load data:
data("primates")
#Create a binary trait factor 
adultMass_binary<-ifelse(primates$data$adultMass > 7350, "big", "small")
adultMass_binary<-as.factor(as.factor(adultMass_binary))
names(adultMass_binary)<-rownames(primates$data)
#Model trait evolution accounting for phylogenetic uncertainty
tree_binary<-tree_discrete(data = adultMass_binary,phy = primates$phy,
model = "ARD",transform = "none",n.tree = 30,n.cores = 2,track = TRUE)
#Print summary statistics
summary(tree_binary)
sensi_plot(tree_binary)
sensi_plot(tree_binary,graphs="q12")
sensi_plot(tree_binary,graphs="q21")
#Use a different evolutionary model or transformation.
tree_binary_lambda<-tree_discrete(data = adultMass_binary,phy = primates$phy,
model = "SYM",transform = "lambda",n.tree = 30,n.cores = 2,track = TRUE)
summary(tree_binary_lambda) #Using Pagel's Lambda
sensi_plot(tree_binary_lambda)  
#Symmetrical rates, with an Early Burst (EB) model of trait evolution
tree_binary_SYM_EB<-tree_discrete(data = adultMass_binary,phy = primates$phy,
model = "SYM",transform = "EB",n.tree = 30,n.cores = 2,track = TRUE)
summary(tree_binary_SYM_EB)
sensi_plot(tree_binary_SYM_EB) 
sensi_plot(tree_binary_SYM_EB,graphs="optpar") 
}
}
\references{
Yang Z. 2006. Computational Molecular Evolution. Oxford University Press: Oxford. 

Harmon Luke J, Jason T Weir, Chad D Brock, Richard E Glor, and Wendell Challenger. 2008.
GEIGER: investigating evolutionary radiations. Bioinformatics 24:129-131.
}
\seealso{
\code{\link[geiger]{fitDiscrete}}
}
\author{
Gijsbert Werner & Caterina Penone
}
