% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paths.R
\name{st_network_paths}
\alias{st_network_paths}
\title{Paths between points in geographical space}
\usage{
st_network_paths(
  x,
  from,
  to = igraph::V(x),
  weights = NULL,
  type = "shortest",
  ...
)
}
\arguments{
\item{x}{An object of class \code{\link{sfnetwork}}.}

\item{from}{The geospatial point from which the paths will be
calculated. Can be an object an object of class \code{\link[sf]{sf}} or
\code{\link[sf]{sfc}}, containing a single feature. When multiple features
are given, only the first one is taken.
Alternatively, it can be an integer, referring to the index of the
node from which the paths will be calculated, or a character,
referring to the name of the node from which the paths will be
calculated.}

\item{to}{The (set of) geospatial point(s) to which the paths will be
calculated. Can be an object of  class \code{\link[sf]{sf}} or
\code{\link[sf]{sfc}}.
Alternatively it can be a numeric vector containing the indices of the nodes
to which the paths will be calculated, or a character vector
containing the names of the nodes to which the paths will be
calculated. By default, all nodes in the network are included.}

\item{weights}{The edge weights to be used in the shortest path calculation.
Can be a numeric vector giving edge weights, or a column name referring to
an attribute column in the edges table containing those weights. If set to
\code{NULL}, the values of a column named \code{weight} in the edges table
will be used automatically, as long as this column is present. If not, the
geographic edge lengths will be calculated internally and used as weights.
If set to \code{NA}, no weights are used, even if the edges have a
\code{weight} column. Ignored when \code{type = 'all_simple'}.}

\item{type}{Character defining which type of path calculation should be
performed. If set to \code{'shortest'} paths are calculated using
\code{\link[igraph]{shortest_paths}}, if set to
\code{'all_shortest'} paths are calculated using
\code{\link[igraph]{all_shortest_paths}}, if set to
\code{'all_simple'} paths are calculated using
\code{\link[igraph]{all_simple_paths}}. Defaults to \code{'shortest'}.}

\item{...}{Arguments passed on to the corresponding
\code{\link[igraph:shortest_paths]{igraph}} or
\code{\link[igraph:all_simple_paths]{igraph}} function. Arguments
\code{predecessors} and \code{inbound.edges} are ignored.}
}
\value{
An object of class \code{\link[tibble]{tbl_df}} with one row per
returned path. Depending on the setting of the \code{type} argument,
columns can be \code{node_paths} (a list column with for each path the
ordered indices of nodes present in that path) and \code{edge_paths}
(a list column with for each path the ordered indices of edges present in
that path). \code{'all_shortest'} and \code{'all_simple'} return only
\code{node_paths}, while \code{'shortest'} returns both.
}
\description{
Combined wrapper around \code{\link[igraph]{shortest_paths}},
\code{\link[igraph]{all_shortest_paths}} and
\code{\link[igraph]{all_simple_paths}} from \code{\link[igraph]{igraph}},
allowing to provide any geospatial point as \code{from} argument and any
set of geospatial points as \code{to} argument. If such a geospatial point
is not equal to a node in the network, it will be snapped to its nearest
node before calculating the shortest or simple paths.
}
\details{
See the \code{\link[igraph:shortest_paths]{igraph}} or
\code{\link[igraph:all_simple_paths]{igraph}} documentation.
}
\examples{
library(sf, quietly = TRUE)
library(tidygraph, quietly = TRUE)

# Create a network with edge lengths as weights.
# These weights will be used automatically in shortest paths calculation.
net = as_sfnetwork(roxel, directed = FALSE) \%>\%
  st_transform(3035) \%>\%
  activate("edges") \%>\%
  mutate(weight = edge_length())

# Providing node indices.
paths = st_network_paths(net, from = 495, to = 121)
paths

node_path = paths \%>\%
  slice(1) \%>\%
  pull(node_paths) \%>\%
  unlist()
node_path

oldpar = par(no.readonly = TRUE)
par(mar = c(1,1,1,1))
plot(net, col = "grey")
plot(slice(activate(net, "nodes"), node_path), col = "red", add = TRUE)
par(oldpar)

# Providing nodes as spatial points.
# Points that don't equal a node will be snapped to their nearest node.
p1 = st_geometry(net, "nodes")[495] + st_sfc(st_point(c(50, -50)))
st_crs(p1) = st_crs(net)
p2 = st_geometry(net, "nodes")[121] + st_sfc(st_point(c(-10, 100)))
st_crs(p2) = st_crs(net)

paths = st_network_paths(net, from = p1, to = p2)
paths

node_path = paths \%>\%
  slice(1) \%>\%
  pull(node_paths) \%>\%
  unlist()
node_path

oldpar = par(no.readonly = TRUE)
par(mar = c(1,1,1,1))
plot(net, col = "grey")
plot(c(p1, p2), col = "black", pch = 8, add = TRUE)
plot(slice(activate(net, "nodes"), node_path), col = "red", add = TRUE)
par(oldpar)

# Using another column for weights.
net \%>\%
  activate("edges") \%>\%
  mutate(foo = runif(n(), min = 0, max = 1)) \%>\%
  st_network_paths(p1, p2, weights = "foo")

# Obtaining all simple paths between two nodes.
# Beware, this function can take long when:
# --> Providing a lot of 'to' nodes.
# --> The network is large and dense.
net = as_sfnetwork(roxel, directed = TRUE)
st_network_paths(net, from = 1, to = 12, type = "all_simple")

# Obtaining all shortest paths between two nodes.
# Not using edge weights.
# Hence, a shortest path is the paths with the least number of edges.
st_network_paths(net, from = 5, to = 1, weights = NA, type = "all_shortest")

}
\seealso{
\code{\link{st_network_cost}}
}
