% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulations.R
\name{MakePositiveDefinite}
\alias{MakePositiveDefinite}
\title{Making positive definite}
\usage{
MakePositiveDefinite(
  omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = NULL,
  scale = TRUE,
  u_list = c(1e-10, 1),
  tol = .Machine$double.eps^0.25
)
}
\arguments{
\item{omega}{input matrix.}

\item{pd_strategy}{method to ensure that the generated precision matrix is
positive definite (and hence can be a covariance matrix). If
\code{pd_strategy="diagonally_dominant"}, the precision matrix is made
diagonally dominant by setting the diagonal entries to the sum of absolute
values on the corresponding row and a constant u. If
\code{pd_strategy="min_eigenvalue"}, diagonal entries are set to the sum of
the absolute value of the smallest eigenvalue of the precision matrix with
zeros on the diagonal and a constant u.}

\item{ev_xx}{expected proportion of explained variance by the first Principal
Component (PC1) of a Principal Component Analysis. This is the largest
eigenvalue of the correlation (if \code{scale=TRUE}) or covariance (if
\code{scale=FALSE}) matrix divided by the sum of eigenvalues. If
\code{ev_xx=NULL} (the default), the constant u is chosen by maximising the
contrast of the correlation matrix.}

\item{scale}{logical indicating if the proportion of explained variance by
PC1 should be computed from the correlation (\code{scale=TRUE}) or
covariance (\code{scale=FALSE}) matrix.}

\item{u_list}{vector with two numeric values defining the range of values to
explore for constant u.}

\item{tol}{accuracy for the search of parameter u as defined in
\code{\link[stats]{optimise}}.}
}
\value{
A list with: \item{omega}{positive definite matrix.} \item{u}{value
  of the constant u.}
}
\description{
Determines the diagonal entries of a symmetric matrix to ensure it is
positive definite. For this, diagonal entries of the matrix are defined to be
higher than (i) the sum of entries on the corresponding row, which ensure it
is diagonally dominant, or (ii) the absolute value of the smallest eigenvalue
of the same matrix with a diagonal of zeros. The magnitude of (standardised)
values in the inverse matrix is tuned by adding a constant u to the diagonal
entries. Considering the matrix to make positive definite is a precision
matrix, the constant u is chosen to (i) maximise the \code{\link{Contrast}}
of the corresponding correlation matrix, or (ii) tune the proportion of
explained variance by the first Principal Component \code{ev_xx} (i.e.
largest eigenvalue of the covariance/correlation matrix divided by the sum of
eigenvalues).
}
\examples{
# Simulation of a symmetric matrix
p <- 5
set.seed(1)
omega <- matrix(rnorm(p * p), ncol = p)
omega <- omega + t(omega)
diag(omega) <- 0

# Diagonal dominance maximising contrast
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant"
)
eigen(omega_pd$omega)$values # positive eigenvalues

# Diagonal dominance with specific proportion of explained variance by PC1
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = 0.55
)
lambda_inv <- eigen(cov2cor(solve(omega_pd$omega)))$values
max(lambda_inv) / sum(lambda_inv) # expected ev

# Version not scaled (using eigenvalues from the covariance)
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "diagonally_dominant",
  ev_xx = 0.55, scale = FALSE
)
lambda_inv <- 1 / eigen(omega_pd$omega)$values
max(lambda_inv) / sum(lambda_inv) # expected ev

# Non-negative eigenvalues maximising contrast
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue"
)
eigen(omega_pd$omega)$values # positive eigenvalues

# Non-negative eigenvalues with specific proportion of explained variance by PC1
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue",
  ev_xx = 0.7
)
lambda_inv <- eigen(cov2cor(solve(omega_pd$omega)))$values
max(lambda_inv) / sum(lambda_inv)

# Version not scaled (using eigenvalues from the covariance)
omega_pd <- MakePositiveDefinite(omega,
  pd_strategy = "min_eigenvalue",
  ev_xx = 0.7, scale = FALSE
)
lambda_inv <- 1 / eigen(omega_pd$omega)$values
max(lambda_inv) / sum(lambda_inv)
}
