% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrinkGPR.R
\name{shrinkGPR}
\alias{shrinkGPR}
\title{Gaussian Process Regression with Shrinkage and Normalizing Flows}
\usage{
shrinkGPR(
  formula,
  data,
  a = 0.5,
  c = 0.5,
  formula_mean,
  a_mean = 0.5,
  c_mean = 0.5,
  sigma2_rate = 10,
  kernel_func = kernel_se,
  n_layers = 10,
  n_latent = 10,
  flow_func = sylvester,
  flow_args,
  n_epochs = 1000,
  auto_stop = TRUE,
  cont_model,
  device,
  display_progress = TRUE,
  optim_control
)
}
\arguments{
\item{formula}{object of class "formula": a symbolic representation of the model for the covariance equation, as in \code{\link{lm}}.
The response variable and covariates are specified here.}

\item{data}{\emph{optional} data frame containing the response variable and the covariates. If not found in \code{data},
the variables are taken from \code{environment(formula)}. No \code{NA}s are allowed in the response variable or covariates.}

\item{a}{positive real number controlling the behavior at the origin of the shrinkage prior for the covariance structure. The default is 0.5.}

\item{c}{positive real number controlling the tail behavior of the shrinkage prior for the covariance structure. The default is 0.5.}

\item{formula_mean}{\emph{optional} formula for the linear mean equation. If provided, the covariates for the mean structure
are specified separately from the covariance structure. A response variable is not required in this formula.}

\item{a_mean}{positive real number controlling the behavior at the origin of the shrinkage for the mean structure. The default is 0.5.}

\item{c_mean}{positive real number controlling the tail behavior of the shrinkage prior for the mean structure. The default is 0.5.}

\item{sigma2_rate}{positive real number controlling the prior rate parameter for the residual variance. The default is 10.}

\item{kernel_func}{function specifying the covariance kernel. The default is \code{\link{kernel_se}}, a squared exponential kernel.
For guidance on how to provide a custom kernel function, see Details.}

\item{n_layers}{positive integer specifying the number of flow layers in the normalizing flow. The default is 10.}

\item{n_latent}{positive integer specifying the dimensionality of the latent space for the normalizing flow. The default is 10.}

\item{flow_func}{function specifying the normalizing flow transformation. The default is \code{\link{sylvester}}.
For guidance on how to provide a custom flow function, see Details.}

\item{flow_args}{\emph{optional} named list containing arguments for the flow function. If not provided, default arguments are used.
For guidance on how to provide a custom flow function, see Details.}

\item{n_epochs}{positive integer specifying the number of training epochs. The default is 1000.}

\item{auto_stop}{logical value indicating whether to enable early stopping based on convergence. The default is \code{TRUE}.}

\item{cont_model}{\emph{optional} object returned from a previous \code{shrinkGPR} call, enabling continuation of training from the saved state.}

\item{device}{\emph{optional} device to run the model on, e.g., \code{torch_device("cuda")} for GPU or \code{torch_device("cpu")} for CPU.
Defaults to GPU if available; otherwise, CPU.}

\item{display_progress}{logical value indicating whether to display progress bars and messages during training. The default is \code{TRUE}.}

\item{optim_control}{\emph{optional} named list containing optimizer parameters. If not provided, default settings are used.}
}
\value{
A list object of class \code{shrinkGPR}, containing:
\item{\code{model}}{The best-performing trained model.}
\item{\code{loss}}{The best loss value (ELBO) achieved during training.}
\item{\code{loss_stor}}{A numeric vector storing the ELBO values at each training iteration.}
\item{\code{last_model}}{The model state at the final iteration.}
\item{\code{optimizer}}{The optimizer object used during training.}
\item{\code{model_internals}}{Internal objects required for predictions and further training, such as model matrices and formulas.}
}
\description{
\code{shrinkGPR} implements Gaussian process regression (GPR) with a hierarchical shrinkage prior for hyperparameter estimation,
incorporating normalizing flows to approximate the posterior distribution. The function facilitates model specification, optimization,
and training, including support for early stopping, user-defined kernels, and flow-based transformations.
}
\details{
This implementation provides a computationally efficient framework for GPR, enabling flexible modeling of mean and covariance structures.
Users can specify custom kernel functions, flow transformations, and hyperparameter configurations to adapt the model to their data.

The \code{shrinkGPR} function combines Gaussian process regression with shrinkage priors and normalizing flows for efficient
and flexible hyperparameter estimation. It supports custom kernels, hierarchical shrinkage priors for mean and covariance structures,
and flow-based posterior approximations. The \code{auto_stop} option allows early stopping based on lack of improvement in ELBO.

\strong{Custom Kernel Functions}

Users can define custom kernel functions for the covariance structure of the Gaussian process by passing them to the \code{kernel_func} argument.
A valid kernel function must follow the same structure as the provided \code{kernel_se} (squared exponential kernel). The function should:

\enumerate{
\item \strong{Accept the following arguments:}
\itemize{
\item \code{thetas}: A \code{torch_tensor} of dimensions \code{n_latent x d}, representing latent length-scale parameters.
\item \code{tau}: A \code{torch_tensor} of length \code{n_latent}, representing latent scaling factors.
\item \code{x}: A \code{torch_tensor} of dimensions \code{N x d}, containing the input data points.
\item \code{x_star}: Either \code{NULL} or a \code{torch_tensor} of dimensions \code{N_new x d}. If \code{NULL}, the kernel is computed for \code{x} against itself.
Otherwise, it computes the kernel between \code{x} and \code{x_star}.
}

\item \strong{Return:}
\itemize{
\item If \code{x_star = NULL}, the function must return a \code{torch_tensor} of dimensions \code{n_latent x N x N}, representing pairwise covariances
between all points in \code{x}.
\item If \code{x_star} is provided, the function must return a \code{torch_tensor} of dimensions \code{n_latent x N_new x N},
representing pairwise covariances between \code{x_star} and \code{x}.
}

\item \strong{Requirements:}
\itemize{
\item The kernel must compute a valid positive semi-definite covariance matrix.
\item It should use efficient tensor operations from the Torch library (e.g., \code{torch_bmm}, \code{torch_sum}) to ensure compatibility with GPUs or CPUs.
}
}

\strong{Testing a Custom Kernel Function}

To test a custom kernel function:
\enumerate{
\item \strong{Verify Dimensions:}
\itemize{
\item When \code{x_star = NULL}, ensure the output is \code{n_latent x N x N}.
\item When \code{x_star} is provided, ensure the output is \code{n_latent x N_new x N}.
}
\item \strong{Check Positive Semi-Definiteness:}
Validate that the kernel produces a positive semi-definite covariance matrix for valid inputs.
\item \strong{Integrate:}
Use the custom kernel with \code{shrinkGPR} to confirm its compatibility.
}

Examples of kernel functions can be found in the \code{kernel_funcs.R} file in the package source code,
which are documented in the \code{\link{kernel_functions}} help file.

\strong{Custom Flow Functions}

Users can define custom flow functions for use in Gaussian process regression models by following the structure
and conventions of the provided \code{sylvester} function. A valid flow function should be implemented as a
\code{nn_module} in \code{torch} and must meet the following requirements:

\strong{Structure of a Custom Flow Function}

\enumerate{
\item \strong{Initialization (\code{initialize})}:
\itemize{
\item Include all required parameters as \code{nn_parameter} or \code{nn_buffer}, and initialize them appropriately.
\item Parameters may include matrices for transformations (e.g., triangular matrices), biases, or other learnable components.
}

\item \strong{Forward Pass (\code{forward})}:
\itemize{
\item The \code{forward} method should accept an input tensor \code{z} of dimensions \code{n_latent x D}.
\item The method must:
\itemize{
\item Compute the transformed tensor \code{z}.
\item Compute the log determinant of the Jacobian (\code{log|det J|}).
}
\item The method should return a list containing:
\itemize{
\item \code{zk}: The transformed samples after applying the flow (\code{n_latent x D}).
\item \code{log_diag_j}: A tensor of size \code{n_latent} containing the log determinant of the Jacobian for each sample.
}
}

\item \strong{Output Dimensions}:
\itemize{
\item Input tensor \code{z}: \code{n_latent x D}.
\item Outputs:
\itemize{
\item \code{zk}: \code{n_latent x D}.
\item \code{log_diag_j}: \code{n_latent}.
}
}
}
An example of a flow function can be found in the \code{sylvester.R} file in the package source code,
which is documented in the \code{\link{sylvester}} help file.
}
\examples{
\donttest{
if (torch::torch_is_installed()) {
  # Simulate data
  set.seed(123)
  torch::torch_manual_seed(123)
  n <- 100
  x <- matrix(runif(n * 2), n, 2)
  y <- sin(2 * pi * x[, 1]) + rnorm(n, sd = 0.1)
  data <- data.frame(y = y, x1 = x[, 1], x2 = x[, 2])

  # Fit GPR model
  res <- shrinkGPR(y ~ x1 + x2, data = data)

  # Check convergence
  plot(res$loss_stor, type = "l", main = "Loss Over Iterations")

  # Check posterior
  samps <- gen_posterior_samples(res, nsamp = 1000)
  boxplot(samps$thetas) # Second theta is pulled towards zero

  # Predict
  x1_new <- seq(from = 0, to = 1, length.out = 100)
  x2_new <- runif(100)
  y_new <- predict(res, newdata = data.frame(x1 = x1_new, x2 = x2_new), nsamp = 2000)

  # Plot
  quants <- apply(y_new, 2, quantile, c(0.025, 0.5, 0.975))
  plot(x1_new, quants[2, ], type = "l", ylim = c(-1.5, 1.5),
        xlab = "x1", ylab = "y", lwd = 2)
  polygon(c(x1_new, rev(x1_new)), c(quants[1, ], rev(quants[3, ])),
        col = adjustcolor("skyblue", alpha.f = 0.5), border = NA)
  points(x[,1], y)
  curve(sin(2 * pi * x), add = TRUE, col = "forestgreen", lwd = 2, lty = 2)



  # Add mean equation
  res2 <- shrinkGPR(y ~ x1 + x2, formula_mean = ~ x1, data = data)
  }
}
}
\author{
Peter Knaus \email{peter.knaus@wu.ac.at}
}
