% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sievePH.R
\name{sievePH}
\alias{sievePH}
\title{Semiparametric Estimation of Coefficients in a Mark-Specific Proportional Hazards
Model with a Multivariate Continuous Mark, Fully Observed in All Failures}
\usage{
sievePH(eventTime, eventInd, mark, tx)
}
\arguments{
\item{eventTime}{a numeric vector specifying the observed right-censored time to the event of interest}

\item{eventInd}{a numeric vector indicating the event of interest (1 if event, 0 if right-censored)}

\item{mark}{either a numeric vector specifying a univariate continuous mark or a data frame specifying a multivariate continuous mark.
No missing values are permitted for subjects with \code{eventInd = 1}. For subjects with \code{eventInd = 0}, the value(s) in \code{mark} should be set to \code{NA}.}

\item{tx}{a numeric vector indicating the treatment group (1 if treatment, 0 if placebo)}
}
\value{
An object of class \code{sievePH} which can be processed by
\code{\link{summary.sievePH}} to obtain or print a summary of the results. An object of class
\code{sievePH} is a list containing the following components:
\itemize{
\item \code{DRcoef}: a numeric vector of estimates of coefficients \eqn{\phi} in the weight function \eqn{g(v, \phi)} in the density ratio model
\item \code{DRlambda}: an estimate of the Lagrange multiplier in the profile score functions for \eqn{\phi} (that arises by profiling out the nuisance parameter)
\item \code{DRconverged}: a logical value indicating whether the estimation procedure in the density ratio model converged
\item \code{logHR}: an estimate of the marginal log hazard ratio from \code{coxph()} in the \code{survival} package
\item \code{cov}: the estimated joint covariance matrix of \code{DRcoef} and \code{logHR}
\item \code{coxphFit}: an object returned by the call of \code{coxph()}
\item \code{nPlaEvents}: the number of events observed in the placebo group
\item \code{nTxEvents}: the number of events observed in the treatment group
\item \code{mark}: the input object
\item \code{tx}: the input object
}
}
\description{
\code{sievePH} implements the semiparametric estimation method of Juraska and Gilbert (2013) for the multivariate mark-
specific hazard ratio in the competing risks failure time analysis framework. It employs (i) the semiparametric
method of maximum profile likelihood estimation in the treatment-to-placebo mark density
ratio model (Qin, 1998) and (ii) the ordinary method of maximum partial likelihood estimation of the overall log hazard ratio in the Cox model.
\code{sievePH} requires that the multivariate mark data are fully observed in all failures.
}
\details{
\code{sievePH} considers data from a randomized placebo-controlled treatment efficacy trial with a time-to-event endpoint.
The parameter of interest, the mark-specific hazard ratio, is the ratio (treatment/placebo) of the conditional mark-specific hazard functions.
It factors as the product of the mark density ratio (treatment/placebo) and the ordinary marginal hazard function ignoring mark data.
The mark density ratio is estimated using the method of Qin (1998), while the marginal hazard ratio is estimated using \code{coxph()} in the \code{survival} package.
Both estimators are consistent and asymptotically normal. The joint asymptotic distribution of the estimators is detailed in Juraska and Gilbert (2013).
}
\examples{
n <- 500
tx <- rep(0:1, each=n/2)
tm <- c(rexp(n/2, 0.2), rexp(n/2, 0.2 * exp(-0.4)))
cens <- runif(n, 0, 15)
eventTime <- pmin(tm, cens, 3)
eventInd <- as.numeric(tm <= pmin(cens, 3))
mark1 <- ifelse(eventInd==1, c(rbeta(n/2, 2, 5), rbeta(n/2, 2, 2)), NA)
mark2 <- ifelse(eventInd==1, c(rbeta(n/2, 1, 3), rbeta(n/2, 5, 1)), NA)

# fit a model with a univariate mark
fit <- sievePH(eventTime, eventInd, mark1, tx)

# fit a model with a bivariate mark
fit <- sievePH(eventTime, eventInd, data.frame(mark1, mark2), tx)

}
\references{
Juraska, M. and Gilbert, P. B. (2013), Mark-specific hazard ratio model with multivariate continuous marks: an application to vaccine efficacy. \emph{Biometrics} 69(2):328–337.

Qin, J. (1998), Inferences for case-control and semiparametric two-sample density ratio models. \emph{Biometrika} 85, 619–630.
}
\seealso{
\code{\link{summary.sievePH}}, \code{\link{plot.summary.sievePH}}, \code{\link{testIndepTimeMark}} and \code{\link{testDensRatioGOF}}
}
