% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitpsd.R
\name{pl_fit}
\alias{pl_fit}
\alias{exp_fit}
\alias{lnorm_fit}
\alias{tpl_fit}
\title{Distribution-fitting functions}
\usage{
pl_fit(dat, xmin = 1)

exp_fit(dat, xmin = 1)

lnorm_fit(dat, xmin = 1)

tpl_fit(dat, xmin = 1)
}
\arguments{
\item{dat}{The set of values to which the distribution are fit}

\item{xmin}{The minimum possible value to consider when fitting the
distribution}
}
\value{
A list containing at list the following components: 

\itemize{ 
   \item{type: }{The type of distribution fitted (as a character string)}
   \item{method: }{The method used for the fit - here, maximum likelihood,
                 'll'}
   \item{ll: }{The negative log likelihood at the estimated parameter values}
   \item{xmin: }{The value of xmin used for the fit}
   \item{npars: }{The number of parameters of the distribution}
 }

Additionnaly, this list may have one or more of the following parameters 
  depending on the type of distribution that has been fitted: 
  \itemize{ 
    \item{plexpo: }{The exponent of the power-law}
    \item{cutoff: }{The rate of truncation, for truncated power law and 
                exponential fits}
    \item{meanlog: }{The mean of the lognormal distribution}
    \item{sdlog: }{The s.d. of the lognormal distribution}
  }
}
\description{
These functions fit parametric distributions to a set of
  discrete values.
}
\details{
These functions will fit distributions to a set of values using 
  maximum-likelihood estimation. In the context of the 'spatialwarnings' 
  package, they are most-often used to fit parametric distributions on patch
  size distributions. As a result, these functions assume that the data 
  contains only integer, strictly positive values. The type of distribution
  depends on the prefix of the function: 'pl' for power-law, 'tpl' for
  truncated power-law, 'lnorm' for lognormal and 'exp' for an exponential
  distribution. 

In the context of distribution-fitting, 'xmin' represents the minimum value 
  that a distribution can take. It is often used to represent the minimum 
  scale at which a power-law model is appropriate (Clauset et al. 2009), and 
  can be estimated on an empirical distribution using
  \code{\link{xmin_estim}}. Again, please note that the fitting procedure 
  assumes here that xmin is equal or grater than one.

Please note that a best effort is made to have the fit converge, but 
  it may sometimes fail when the parameters are far from their usual 
  range. It is good practice to make sure the fits are sensible when 
  convergence warnings are reported.

For reference, the shape of the distributions is as follow: 

\itemize{
  \item{power-law }{\eqn{x^{-a}}{x^(-a)} where a is the power-law exponent}
  \item{exponential }{\eqn{exp(-bx)}{exp(-bx)} where b is the truncation rate
          of the exponential } 
  \item{truncated power-law }{\eqn{x^{-a}exp(-bx)}{x^(-a)exp(-bx)} where a
    and b are the exponent of the power law and the rate of truncation}
}

The lognormal form follows the \link[=dlnorm]{standard definition}.
}
\examples{

# Fit an exponential model to patch size distribution 
exp_fit(patchsizes(forestgap[[8]]))
 
# Use the estimated parameters as an indicator function
\dontrun{

get_truncation <- function(mat) { 
   c(exp_cutoff = exp_fit(patchsizes(mat))$cutoff)
}
trunc_indic <- compute_indicator(forestgap, get_truncation)
plot(trunc_indic)
plot(indictest(trunc_indic, nulln = 19))

}

}
\references{
Clauset, Aaron, Cosma Rohilla Shalizi, and M. E. J. Newman. 2009. “Power-Law
Distributions in Empirical Data.” SIAM Review 51 (4): 661–703. 
https://doi.org/10.1137/070710111.
}
\seealso{
\code{\link{patchdistr_sews}}, \code{\link{xmin_estim}}
}
