#ifndef STAN_MATH_OPENCL_PRIM_GUMBEL_LCDF_HPP
#define STAN_MATH_OPENCL_PRIM_GUMBEL_LCDF_HPP
#ifdef STAN_OPENCL

#include <stan/math/prim/meta.hpp>
#include <stan/math/prim/err.hpp>
#include <stan/math/prim/fun/constants.hpp>
#include <stan/math/prim/fun/elt_divide.hpp>
#include <stan/math/prim/fun/elt_multiply.hpp>
#include <stan/math/opencl/kernel_generator.hpp>
#include <stan/math/prim/functor/operands_and_partials.hpp>

namespace stan {
namespace math {

/** \ingroup opencl
 * Returns the Gumbel log complementary cumulative distribution function for the
 * given location, and scale. If given containers of matching sizes returns the
 * product of probabilities.
 *
 * @tparam T_y_cl type of scalar outcome
 * @tparam T_loc_cl type of location
 * @tparam T_scale_cl type of scale
 * @param y (Sequence of) scalar(s).
 * @param mu (Sequence of) location(s).
 * @param beta (Sequence of) scale(s).
 * @return The sum of log complementary cumulative probabilities.
 */
template <
    typename T_y_cl, typename T_loc_cl, typename T_scale_cl,
    require_all_prim_or_rev_kernel_expression_t<T_y_cl, T_loc_cl,
                                                T_scale_cl>* = nullptr,
    require_any_not_stan_scalar_t<T_y_cl, T_loc_cl, T_scale_cl>* = nullptr>
return_type_t<T_y_cl, T_loc_cl, T_scale_cl> gumbel_lcdf(
    const T_y_cl& y, const T_loc_cl& mu, const T_scale_cl& beta) {
  static const char* function = "gumbel_lcdf(OpenCL)";
  using T_partials_return = partials_return_t<T_y_cl, T_loc_cl, T_scale_cl>;
  using std::isfinite;
  using std::isnan;

  check_consistent_sizes(function, "Random variable", y, "Location parameter",
                         mu, "Scale parameter", beta);
  const size_t N = max_size(y, mu, beta);
  if (N == 0) {
    return 1.0;
  }

  const auto& y_col = as_column_vector_or_scalar(y);
  const auto& mu_col = as_column_vector_or_scalar(mu);
  const auto& beta_col = as_column_vector_or_scalar(beta);

  const auto& y_val = value_of(y_col);
  const auto& mu_val = value_of(mu_col);
  const auto& beta_val = value_of(beta_col);

  auto check_y_not_nan
      = check_cl(function, "Random variable", y_val, "not NaN");
  auto y_not_nan_expr = !isnan(y_val);
  auto check_mu_finite
      = check_cl(function, "Location parameter", mu_val, "finite");
  auto mu_finite_expr = isfinite(mu_val);
  auto check_beta_positive
      = check_cl(function, "Scale parameter", beta_val, "positive");
  auto beta_positive_expr = 0.0 < beta_val;

  auto scaled_diff = elt_divide(mu_val - y_val, beta_val);
  auto exp_scaled_diff = exp(scaled_diff);
  auto lcdf_expr = colwise_sum(exp_scaled_diff);
  auto y_deriv = elt_divide(exp_scaled_diff, beta_val);
  auto mu_deriv = -y_deriv;
  auto beta_deriv = elt_multiply(y_deriv, scaled_diff);

  matrix_cl<double> lcdf_cl;
  matrix_cl<double> y_deriv_cl;
  matrix_cl<double> mu_deriv_cl;
  matrix_cl<double> beta_deriv_cl;

  results(check_y_not_nan, check_mu_finite, check_beta_positive, lcdf_cl,
          y_deriv_cl, mu_deriv_cl, beta_deriv_cl)
      = expressions(y_not_nan_expr, mu_finite_expr, beta_positive_expr,
                    lcdf_expr, calc_if<!is_constant<T_y_cl>::value>(y_deriv),
                    calc_if<!is_constant<T_loc_cl>::value>(mu_deriv),
                    calc_if<!is_constant<T_scale_cl>::value>(beta_deriv));

  T_partials_return lcdf = -sum(from_matrix_cl(lcdf_cl));

  operands_and_partials<decltype(y_col), decltype(mu_col), decltype(beta_col)>
      ops_partials(y_col, mu_col, beta_col);

  if (!is_constant<T_y_cl>::value) {
    ops_partials.edge1_.partials_ = std::move(y_deriv_cl);
  }
  if (!is_constant<T_loc_cl>::value) {
    ops_partials.edge2_.partials_ = std::move(mu_deriv_cl);
  }
  if (!is_constant<T_scale_cl>::value) {
    ops_partials.edge3_.partials_ = std::move(beta_deriv_cl);
  }
  return ops_partials.build(lcdf);
}

}  // namespace math
}  // namespace stan
#endif
#endif
