% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithms_rna.R
\name{rna_one_sided}
\alias{rna_one_sided}
\title{Recursive Neyman Algorithm for Optimal Sample Allocation in Stratified
Sampling Schemes}
\usage{
rna_one_sided(n, a, bounds = NULL, upper = TRUE)
}
\arguments{
\item{n}{(\code{number})\cr total sample size. A strictly positive scalar.}

\item{a}{(\code{numeric})\cr parameters \eqn{a_1, ..., a_H} of variance function
\eqn{D}. Strictly positive numbers.}

\item{bounds}{(\code{numeric} or \code{NULL}) \cr optional lower or upper bounds
constraints imposed on strata sample sizes. If \code{bounds} is not \code{NULL}, it
is required that \code{n <= sum(bounds)} in case of \code{upper = TRUE}, and
\code{n >= sum(bounds)}, in case of \code{upper = FALSE}.
Strictly positive numbers.}

\item{upper}{(\code{flag}) \cr should values of \code{bounds} be treated as one-sided
upper bounds constraints (default)? Otherwise, they are treated as lower
bounds.}
}
\value{
Numeric vector with optimal sample allocations in strata.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Internal function that implements the recursive Neyman optimal allocation
algorithms, \code{rNa} and \code{LrNa}, described in Wesołowski et al. (2021) and
Wójciak (2022) respectively. The \code{rna_one_sided()} should not be used
directly. Instead, user function \code{\link[=dopt]{dopt()}} or \code{\link[=nopt]{nopt()}} should be used. \cr

The \code{rna_one_sided()} function computes
\deqn{argmin D(x_1,...,x_H) = a^2_1/x_1 + ... + a^2_H/x_H - b,}
under the equality constraint imposed on total sample size
\deqn{x_1 + ... + x_H = n.}
Here, \eqn{H} denotes total number of strata, \eqn{x_1, ..., x_H} are the
strata sample sizes, and \eqn{n > 0}, \eqn{b},
\eqn{a_w > 0, w = 1, ..., H}, are given numbers. \cr
Optionally, one of the following set of one-sided inequality constraints
can be added
\deqn{x_w <= M_w,  w = 1,...,H, (Case I)}
or
\deqn{x_w >= m_w, w = 1,...,H, (Case II)}
where \eqn{m_w > 0} and \eqn{M_w > 0, w = 1, ..., H} are lower and upper
bounds respectively, imposed on sample strata sizes. \cr

User of \code{rna_one_sided()} can choose whether the inequality constraints
will be added to the optimization problem or not. This is achieved with the
proper use of \code{bounds} and \code{upper} arguments of the function. In case of no
inequality constraints to be added, \code{bounds} must be specified as \code{NULL}
(default).
If any bounds should be imposed on sample strata sizes, user must specify
these with \code{bounds} argument.
For the Case I of the upper bounds, \code{upper} flag must be set to \code{TRUE}
(default) and then the \code{rna_one_sided()} performs the \code{rNa}.
For the Case II of lower bounds, \code{upper} flag must be set to \code{FALSE} and
then the \code{rna_one_sided()} performs the \code{LrNa}.
The \code{upper} flag is ignored when \code{bounds} is \code{NULL}.
If no inequality constraints are added, the allocation is given as a closed
form expression, known as Neyman allocation
\deqn{x_w = a_w * n / (a_1 + ... + a_H), w = 1, ..., H.}
}
\note{
For simple random sampling without replacement design in each stratum,
parameters of the variance function \eqn{D} are
\eqn{b = N_1 * S_1^2 + ... + N_H * S_H^2}, and \eqn{a_w = N_w * S_w}, where
\eqn{N_w, S_w, w = 1, ..., H}, are strata sizes and standard deviations of
a study variable in strata respectively. \cr

The \code{rNa} and \code{LrNa} are kind of more general versions of popular recursive
Neyman allocation procedure that is described in Remark 12.7.1 in Sarndal
et al. (1992). It is a procedure of optimal sample allocation in stratified
sampling scheme with simple random sampling without replacement design in
each stratum while not exceeding strata sizes.
}
\examples{
a <- c(3000, 4000, 5000, 2000)
bounds <- c(100, 90, 70, 80)
rna_one_sided(n = 190, a = a, bounds = bounds)
}
\references{
Wesołowski, J., Wieczorkowski, R., Wójciak, W. (2021),
Optimality of the recursive Neyman allocation,
\emph{Journal of Survey Statistics and Methodology},
\doi{10.1093/jssam/smab018},
\doi{10.48550/arXiv.2105.14486} \cr

Wójciak, W. (2022),
Minimum sample size allocation in stratified sampling under constraints on
variance and strata sample sizes,
\doi{10.48550/arXiv.2204.04035} \cr

Sarndal, C.-E., Swensson, B., and Wretman, J. (1992),
\emph{Model Assisted Survey Sampling}, New York, NY: Springer.
}
\seealso{
\code{\link[=dopt]{dopt()}}, \code{\link[=nopt]{nopt()}}, \code{\link[=sga]{sga()}}, \code{\link[=sgaplus]{sgaplus()}}, \code{\link[=coma]{coma()}}.
}
