% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geo_cov_survey_scheme.R
\name{geo_cov_survey_scheme}
\alias{geo_cov_survey_scheme}
\title{Geographic coverage survey scheme}
\usage{
geo_cov_survey_scheme(
  site_data,
  cost_column,
  survey_budget,
  locked_in_column = NULL,
  locked_out_column = NULL,
  exclude_locked_out = FALSE,
  solver = "auto",
  verbose = FALSE
)
}
\arguments{
\item{site_data}{\code{\link[sf:sf]{sf::sf()}} object containing the candidate survey
sites.}

\item{cost_column}{\code{character} name of the column in the argument to
the argument to \code{site_data} that contains the cost for surveying each
site. No missing (\code{NA}) values are permitted.}

\item{survey_budget}{\code{numeric} vector of maximum budgets for the survey
schemes. No missing (\code{NA}) values are permitted.}

\item{locked_in_column}{\code{character} (optional) name of the column in
the argument to \code{site_data} that contains \code{logical}
(\code{TRUE}/ \code{FALSE}) values indicating if certain sites should be
locked into the survey scheme.
No missing (\code{NA}) values are permitted.
Defaults to \code{NULL} such that no sites are locked in.}

\item{locked_out_column}{\code{character} (optional) name of the column in
the argument to \code{site_data} that contains \code{logical}
(\code{TRUE}/ \code{FALSE}) values indicating if certain sites should be
locked out of the survey scheme.
No missing (\code{NA}) values are permitted.
Defaults to \code{NULL} such that no sites are locked out.}

\item{exclude_locked_out}{\code{logical} should locked out planning units
be entirely excluded from the optimization process?
Defaults to \code{FALSE}.}

\item{solver}{\code{character} name of the optimization solver to use
for generating survey schemes.
Available options include: \code{"Rsymphony"}, \code{"gurobi"} and \code{"auto"}.
The \code{"auto"} method will use the Gurobi optimization software if
it is available; otherwise, it will use the SYMPHONY software
via the \pkg{Rsymphony} package.
Defaults to \code{"auto"}.}

\item{verbose}{\code{logical} indicating if information should be
printed while generating survey scheme(s). Defaults to \code{FALSE}.}
}
\value{
\code{matrix} of \code{logical} (\code{TRUE}/ \code{FALSE})
values indicating if a site is selected in a scheme or not. Columns
correspond to sites, and rows correspond to different schemes.
}
\description{
Generate a survey scheme by maximizing the geographic coverage
of surveys.
}
\details{
The integer programming formulation of the \emph{p}-Median
problem (Daskin & Maass 2015) is used to generate survey schemes.
}
\section{Solver}{

This function can use the \pkg{Rsymphony} package and
the \href{https://www.gurobi.com/}{Gurobi optimization software} to generate
survey schemes. Although the \pkg{Rsymphony} package
is easier to install because it is freely available on the
The Comprehensive R Archive Network (CRAN), it is strongly recommended to
install the \href{https://www.gurobi.com/}{Gurobi optimization software} and the
\pkg{gurobi} R package because it can generate survey schemes much faster.
Note that special academic licenses are available at no cost.
Installation instructions are available online for \href{https://www.gurobi.com/documentation/9.1/quickstart_linux/r_ins_the_r_package.html}{Linux}, \href{https://www.gurobi.com/documentation/9.1/quickstart_windows/r_ins_the_r_package.html}{Windows}, and \href{https://www.gurobi.com/documentation/9.1/quickstart_mac/r_ins_the_r_package.html}{Mac OS} operating systems.
}

\examples{
# set seed for reproducibility
set.seed(123)

# simulate data
 x <- sf::st_as_sf(
   tibble::tibble(x = rnorm(4), y = rnorm(4),
                  v1 = c(0.1, 0.2, 0.3, 10), # environmental axis 1
                  v2 = c(0.1, 0.2, 0.3, 10), # environmental axis 2
                  cost = rep(1, 4)),
    coords = c("x", "y"))

# plot the sites' locations
plot(st_geometry(x), pch = 16, cex = 3)

# generate scheme with a budget of 2
s <- geo_cov_survey_scheme(x, "cost", 2)

# print scheme
print(s)

# plot scheme
x$scheme <- c(s)
plot(x[, "scheme"], pch = 16, cex = 3)
}
\references{
Daskin MS & Maass KL (2015) The p-median problem. In \emph{Location Science}
(pp. 21-45). Springer, Cham.
}
