% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_target.R
\name{tar_target}
\alias{tar_target}
\title{Declare a target.}
\usage{
tar_target(
  name,
  command,
  pattern = NULL,
  tidy_eval = targets::tar_option_get("tidy_eval"),
  packages = targets::tar_option_get("packages"),
  library = targets::tar_option_get("library"),
  format = targets::tar_option_get("format"),
  iteration = targets::tar_option_get("iteration"),
  error = targets::tar_option_get("error"),
  memory = targets::tar_option_get("memory"),
  garbage_collection = targets::tar_option_get("garbage_collection"),
  deployment = targets::tar_option_get("deployment"),
  priority = targets::tar_option_get("priority"),
  resources = targets::tar_option_get("resources"),
  storage = targets::tar_option_get("storage"),
  retrieval = targets::tar_option_get("retrieval"),
  cue = targets::tar_option_get("cue")
)
}
\arguments{
\item{name}{Symbol, name of the target. A target
name must be a valid name for a symbol in R, and it
must not start with a dot. Subsequent targets
can refer to this name symbolically to induce a dependency relationship:
e.g. \code{tar_target(downstream_target, f(upstream_target))} is a
target named \code{downstream_target} which depends on a target
\code{upstream_target} and a function \code{f()}. In addition, a target's
name determines its random number generator seed. In this way,
each target runs with a reproducible seed so someone else
running the same pipeline should get the same results,
and no two targets in the same pipeline share the same seed.
(Even dynamic branches have different names and thus different seeds.)
You can recover the seed of a completed target
with \code{tar_meta(your_target, seed)} and run \code{set.seed()} on the result
to locally recreate the target's initial RNG state.}

\item{command}{R code to run the target.}

\item{pattern}{Language to define branching for a target.
For example, in a pipeline with numeric vector targets \code{x} and \code{y},
\code{tar_target(z, x + y, pattern = map(x, y))} implicitly defines
branches of \code{z} that each compute \code{x[1] + y[1]}, \code{x[2] + y[2]},
and so on. See the user manual for details.}

\item{tidy_eval}{Logical, whether to enable tidy evaluation
when interpreting \code{command} and \code{pattern}. If \code{TRUE}, you can use the
"bang-bang" operator \verb{!!} to programmatically insert
the values of global objects.}

\item{packages}{Character vector of packages to load right before
the target builds. Use \code{tar_option_set()} to set packages
globally for all subsequent targets you define.}

\item{library}{Character vector of library paths to try
when loading \code{packages}.}

\item{format}{Optional storage format for the target's return value.
With the exception of \code{format = "file"}, each target
gets a file in \verb{_targets/objects}, and each format is a different
way to save and load this file. See the "Storage formats" section
for a detailed list of possible data storage formats.}

\item{iteration}{Character of length 1, name of the iteration mode
of the target. Choices:
\itemize{
\item \code{"vector"}: branching happens with \code{vctrs::vec_slice()} and
aggregation happens with \code{vctrs::vec_c()}.
\item \code{"list"}, branching happens with \verb{[[]]} and aggregation happens with
\code{list()}.
\item \code{"group"}: \code{dplyr::group_by()}-like functionality to branch over
subsets of a data frame. The target's return value must be a data
frame with a special \code{tar_group} column of consecutive integers
from 1 through the number of groups. Each integer designates a group,
and a branch is created for each collection of rows in a group.
See the \code{\link[=tar_group]{tar_group()}} function to see how you can
create the special \code{tar_group} column with \code{dplyr::group_by()}.
}}

\item{error}{Character of length 1, what to do if the target
stops and throws an error. Options:
\itemize{
\item \code{"stop"}: the whole pipeline stops and throws an error.
\item \code{"continue"}: the whole pipeline keeps going.
\item \code{"abridge"}: any currently running targets keep running,
but no new targets launch after that.
(Visit \url{https://books.ropensci.org/targets/debugging.html}
to learn how to debug targets using saved workspaces.)
}}

\item{memory}{Character of length 1, memory strategy.
If \code{"persistent"}, the target stays in memory
until the end of the pipeline (unless \code{storage} is \code{"worker"},
in which case \code{targets} unloads the value from memory
right after storing it in order to avoid sending
copious data over a network).
If \code{"transient"}, the target gets unloaded
after every new target completes.
Either way, the target gets automatically loaded into memory
whenever another target needs the value.
For cloud-based dynamic files such as \code{format = "aws_file"},
this memory strategy applies to
temporary local copies of the file in \verb{_targets/scratch/"}:
\code{"persistent"} means they remain until the end of the pipeline,
and \code{"transient"} means they get deleted from the file system
as soon as possible. The former conserves bandwidth,
and the latter conserves local storage.}

\item{garbage_collection}{Logical, whether to run \code{base::gc()}
just before the target runs.}

\item{deployment}{Character of length 1, only relevant to
\code{\link[=tar_make_clustermq]{tar_make_clustermq()}} and \code{\link[=tar_make_future]{tar_make_future()}}. If \code{"worker"},
the target builds on a parallel worker. If \code{"main"},
the target builds on the host machine / process managing the pipeline.}

\item{priority}{Numeric of length 1 between 0 and 1. Controls which
targets get deployed first when multiple competing targets are ready
simultaneously. Targets with priorities closer to 1 get built earlier
(and polled earlier in \code{\link[=tar_make_future]{tar_make_future()}}).}

\item{resources}{Object returned by \code{tar_resources()}
with optional settings for high-performance computing
functionality, alternative data storage formats,
and other optional capabilities of \code{targets}.
See \code{tar_resources()} for details.}

\item{storage}{Character of length 1, only relevant to
\code{\link[=tar_make_clustermq]{tar_make_clustermq()}} and \code{\link[=tar_make_future]{tar_make_future()}}.
Must be one of the following values:
\itemize{
\item \code{"main"}: the target's return value is sent back to the
host machine and saved/uploaded locally.
\item \code{"worker"}: the worker saves/uploads the value.
\item \code{"none"}: almost never recommended. It is only for
niche situations, e.g. the data needs to be loaded
explicitly from another language. If you do use it,
then the return value of the target is totally ignored
when the target ends, but
each downstream target still attempts to load the data file
(except when \code{retrieval = "none"}).

If you select \code{storage = "none"}, then
the return value of the target's command is ignored,
and the data is not saved automatically.
As with dynamic files (\code{format = "file"} or \code{"aws_file"}) it is the
responsibility of the user to write to
\code{\link[=tar_path]{tar_path()}} from inside the target.
An example target
could look something like
tar_target(x,
{saveRDS("value", tar_path(create_dir = TRUE)); "ignored"},
storage = "none")`.

The distinguishing feature of \code{storage = "none"}
(as opposed to \code{format = "file"} or \code{"aws_file"})
is that in the general case,
downstream targets will automatically try to load the data
from the data store as a dependency. As a corollary, \code{storage = "none"}
is completely unnecessary if \code{format} is \code{"file"} or \code{"aws_file"}.
}}

\item{retrieval}{Character of length 1, only relevant to
\code{\link[=tar_make_clustermq]{tar_make_clustermq()}} and \code{\link[=tar_make_future]{tar_make_future()}}.
Must be one of the following values:
\itemize{
\item \code{"main"}: the target's dependencies are loaded on the host machine
and sent to the worker before the target builds.
\item \code{"worker"}: the worker loads the targets dependencies.
\item \code{"none"}: the dependencies are not loaded at all.
This choice is almost never recommended. It is only for
niche situations, e.g. the data needs to be loaded
explicitly from another language.
}}

\item{cue}{An optional object from \code{tar_cue()} to customize the
rules that decide whether the target is up to date.}
}
\value{
A target object. Users should not modify these directly,
just feed them to \code{\link[=list]{list()}} in your target script file
(default: \verb{_targets.R}).
}
\description{
A target is a single step of computation in a pipeline.
It runs an R command and returns a value.
This value gets treated as an R object that can be used
by the commands of targets downstream. Targets that
are already up to date are skipped. See the user manual
for more details.
}
\section{Target objects}{

Functions like \code{tar_target()} produce target objects,
special objects with specialized sets of S3 classes.
Target objects represent skippable steps of the analysis pipeline
as described at \url{https://books.ropensci.org/targets/}.
Please read the walkthrough at
\url{https://books.ropensci.org/targets/walkthrough.html}
to understand the role of target objects in analysis pipelines.

For developers,
\url{https://wlandau.github.io/targetopia/contributing.html#target-factories}
explains target factories (functions like this one which generate targets)
and the design specification at
\url{https://books.ropensci.org/targets-design/}
details the structure and composition of target objects.
}

\section{Storage formats}{

\itemize{
\item \code{"rds"}: Default, uses \code{saveRDS()} and \code{readRDS()}. Should work for
most objects, but slow.
\item \code{"qs"}: Uses \code{qs::qsave()} and \code{qs::qread()}. Should work for
most objects, much faster than \code{"rds"}. Optionally set the
preset for \code{qsave()} through \code{tar_resources()} and \code{tar_resources_qs()}.
\item \code{"feather"}: Uses \code{arrow::write_feather()} and
\code{arrow::read_feather()} (version 2.0). Much faster than \code{"rds"},
but the value must be a data frame. Optionally set
\code{compression} and \code{compression_level} in \code{arrow::write_feather()}
through \code{tar_resources()} and \code{tar_resources_feather()}.
Requires the \code{arrow} package (not installed by default).
\item \code{"parquet"}: Uses \code{arrow::write_parquet()} and
\code{arrow::read_parquet()} (version 2.0). Much faster than \code{"rds"},
but the value must be a data frame. Optionally set
\code{compression} and \code{compression_level} in \code{arrow::write_parquet()}
through \code{tar_resources()} and \code{tar_resources_parquet()}.
Requires the \code{arrow} package (not installed by default).
\item \code{"fst"}: Uses \code{fst::write_fst()} and \code{fst::read_fst()}.
Much faster than \code{"rds"}, but the value must be
a data frame. Optionally set the compression level for
\code{fst::write_fst()} through \code{tar_resources()} and \code{tar_resources_fst()}.
Requires the \code{fst} package (not installed by default).
\item \code{"fst_dt"}: Same as \code{"fst"}, but the value is a \code{data.table}.
Optionally set the compression level the same way as for \code{"fst"}.
\item \code{"fst_tbl"}: Same as \code{"fst"}, but the value is a \code{tibble}.
Optionally set the compression level the same way as for \code{"fst"}.
\item \code{"keras"}: Uses \code{keras::save_model_hdf5()} and
\code{keras::load_model_hdf5()}. The value must be a Keras model.
Requires the \code{keras} package (not installed by default).
\item \code{"torch"}: Uses \code{torch::torch_save()} and \code{torch::torch_load()}.
The value must be an object from the \code{torch} package
such as a tensor or neural network module.
Requires the \code{torch} package (not installed by default).
\item \code{"file"}: A dynamic file. To use this format,
the target needs to manually identify or save some data
and return a character vector of paths
to the data. (These paths must be existing files
and nonempty directories.)
Then, \code{targets} automatically checks those files and cues
the appropriate build decisions if those files are out of date.
Those paths must point to files or directories,
and they must not contain characters \code{|} or \code{*}.
All the files and directories you return must actually exist,
or else \code{targets} will throw an error. (And if \code{storage} is \code{"worker"},
\code{targets} will first stall out trying to wait for the file
to arrive over a network file system.)
If the target does not create any files, the return value should be
\code{character(0)}.
\item \code{"url"}: A dynamic input URL. It works like \code{format = "file"}
except the return value of the target is a URL that already exists
and serves as input data for downstream targets. Optionally
supply a custom \code{curl} handle through
\code{tar_resources()} and \code{tar_resources_url()}.
in \code{new_handle()}, \code{nobody = TRUE} is important because it
ensures \code{targets} just downloads the metadata instead of
the entire data file when it checks time stamps and hashes.
The data file at the URL needs to have an ETag or a Last-Modified
time stamp, or else the target will throw an error because
it cannot track the data. Also, use extreme caution when
trying to use \code{format = "url"} to track uploads. You must be absolutely
certain the ETag and Last-Modified time stamp are fully updated
and available by the time the target's command finishes running.
\code{targets} makes no attempt to wait for the web server.
\item \code{"aws_rds"}, \code{"aws_qs"}, \code{"aws_parquet"}, \code{"aws_fst"}, \code{"aws_fst_dt"},
\code{"aws_fst_tbl"}, \code{"aws_keras"}: versions of the
respective formats \code{"rds"}, \code{"qs"}, etc. powered by
Amazon Web Services (AWS) Simple Storage Service (S3).
The only difference is that the data file is
uploaded to the AWS S3 bucket
you supply to \code{tar_resources_aws()}. See the cloud computing chapter
of the manual for details.
\item \code{"aws_file"}: arbitrary dynamic files on AWS S3. The target
should return a path to a temporary local file, then
\code{targets} will automatically upload this file to an S3
bucket and track it for you. Unlike \code{format = "file"},
\code{format = "aws_file"} can only handle one single file,
and that file must not be a directory.
\code{\link[=tar_read]{tar_read()}} and downstream targets
download the file to \verb{_targets/scratch/} locally and return the path.
\verb{_targets/scratch/} gets deleted at the end of \code{\link[=tar_make]{tar_make()}}.
Requires the same \code{resources} and other configuration details
as the other AWS-powered formats. See the cloud computing
chapter of the manual for details.
\item An entirely custom specification produced by \code{\link[=tar_format]{tar_format()}}.
}
}

\examples{
# Defining targets does not run them.
data <- tar_target(target_name, get_data(), packages = "tidyverse")
analysis <- tar_target(analysis, analyze(x), pattern = map(x))
# Pipelines accept targets.
pipeline <- list(data, analysis)
# Tidy evaluation
tar_option_set(envir = environment())
n_rows <- 30L
data <- tar_target(target_name, get_data(!!n_rows))
print(data)
# Disable tidy evaluation:
data <- tar_target(target_name, get_data(!!n_rows), tidy_eval = FALSE)
print(data)
tar_option_reset()
# In a pipeline:
if (identical(Sys.getenv("TAR_EXAMPLES"), "true")) {
tar_dir({ # tar_dir() runs code from a temporary directory.
tar_script(tar_target(x, 1 + 1), ask = FALSE)
tar_make()
tar_read(x)
})
}
}
\seealso{
Other targets: 
\code{\link{tar_cue}()},
\code{\link{tar_format}()},
\code{\link{tar_target_raw}()}
}
\concept{targets}
