% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictive.R
\name{predict.ithresh}
\alias{predict.ithresh}
\title{Predictive inference for the largest value observed in N years.}
\usage{
\method{predict}{ithresh}(
  object,
  npy = NULL,
  n_years = 100,
  which_u = c("best", "all"),
  which_v = 1L,
  u_prior = rep(1, length(object$u_vec)),
  type = c("p", "d", "q", "i", "r"),
  hpd = FALSE,
  x = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{"ithresh"}, a result of a call to
\code{\link{ithresh}}.}

\item{npy}{A numeric scalar. The mean number of observations per year
of data, after excluding any missing values, i.e. the number of
non-missing observations divided by total number of years of non-missing
data.}

\item{n_years}{A numeric vector. Value(s) of N.  If \code{which_u = "all"}
then \code{n_years} must have length one.}

\item{which_u}{Either a character scalar or a numeric scalar.
  If \code{which_u} is a character scalar it must be either "best" or "all".

  If \code{which_u = "best"} then the single threshold achieving the largest
  measure of predictive performance in \code{object$pred_perf}, based
  on the validation threshold selected using \code{which_v}, is used to
  perform prediction.  See \code{\link{summary.ithresh}} to print the
  best thresholds for each validation threshold.

  If \code{which_u = "all"} then \emph{all} the thresholds are used to
  perform prediction.  The inferences from each threshold are weighted
  according to the \emph{posterior threshold weights} given in
  equation (15) of Northrop et al. (2017) based on the prior probabilities
  of thresholds in \code{u_prior} and column \code{which_v} of the measures
  of predictive performance in \code{object$pred_perf}.

  Otherwise, \code{which_u} is a numeric scalar that indicates which
  element of \code{object$u_vec} the user wishes to select as a single
  threshold on which to base prediction, that is, \code{which_u} must
  be an integer in {1, ..., \code{length(object$u_vec)}}.}

\item{which_v}{A numeric scalar. Indicates which element of
\code{object$v_vec} is used in selecting a single threshold
(if \code{which_u = "best"}) or weighting the inferences from
all thresholds (if \code{which_u = "all"}).
Note: the default, \code{which_v = 1} gives the \emph{lowest} of the
validation thresholds in \code{object$v_vec}.}

\item{u_prior}{A numeric vector.  Prior probabilities for the training
  thresholds in \code{object$u_vec}.  Only used if \code{which_u = "all"}.

  Only the first
  \code{length(object$u_vec) - length(object$v_vec) + which_v}
  elements of \code{u_prior} are used.  This is because only training
  thresholds up to and including \code{object$v_vec[which_v]} are relevant.
  \code{u_prior} must have length \code{length(object$u_vec)} or
  \code{length(object$u_vec) - length(object$v_vec) + which_v}.

  If \code{u_prior} is not supplied then all (relevant) training thresholds
  are given equal prior probability.
  \code{u_prior} is normalized to have sum equal to 1 inside
  \code{predict.ithresh}.}

\item{type}{A character vector.
  Passed to \code{\link[revdbayes]{predict.evpost}}.
  Indicates which type of inference is required:
\itemize{
  \item "p" for the predictive distribution function,
  \item "d" for the predictive density function,
  \item "q" for the predictive quantile function,
  \item "i" for predictive intervals (see \code{...} to set \code{level}),
  \item "r" for random generation from the predictive distribution.
}
  If \code{which_u = "all"} then only \code{type = "p"} or \code{type = "d"}
  are allowed.}

\item{hpd}{A logical scalar.  The argument \code{hpd} of
\code{\link[revdbayes]{predict.evpost}}. Only relevant if
\code{type = "i"}.}

\item{x}{A numeric vector.  The argument \code{x} of
\code{\link[revdbayes]{predict.evpost}}.  In the current context this
must be a vector (not a matrix, as suggested by the documentation of
\code{\link[revdbayes]{predict.evpost}}).  If \code{x} is not supplied
then a default value is set within
\code{\link[revdbayes]{predict.evpost}}.}

\item{...}{Additional arguments to be passed to
\code{\link[revdbayes]{predict.evpost}}.  In particular:
\code{level}, which can be used to set (multiple) levels
for predictive intervals when \code{type = "i"};
\code{lower_tail} (relevant when \code{type = "p"} or \code{"q"}) and
\code{log} (relevant when \code{type = "d"}).}
}
\value{
An list object of class \code{"ithreshpred"} with a similar
  structure to an object of class \code{"evpred"} returned from
  \code{\link[revdbayes]{predict.evpost}} is returned \emph{invisibly}.
  In addition, the object contains
  \code{u_vec = object$u_vec} and \code{v_vec = object$v_vec},
  \code{which_v} and the index \code{best_u} in
  \code{u_vec = object$u_vec} of the best training threshold based on
  the value of \code{which_v}.
  It also contains the value of the Box-Cox transformation parameter
  \code{lambda}.  This will always be equal to 1 if \code{object} was
  returned from \code{ithresh}.

  If \code{which_u = "all"} then
\itemize{
  \item the list also contains the \emph{posterior threshold weights}
    in component \code{post_thresh_wts}
  \item the component \code{y} is a matrix with \code{length{x}} rows
    and 1 + \code{length(object$u_vec) - length(object$v_vec) + which_v}
    columns.  Column 1 contains the estimated predictive distribution
    function (\code{type = "p"}) or density function (\code{type = "d"})
    obtained using a weighted average of the inferences over different
    training thresholds.  The other columns contain the estimated
    functions for each of the training thresholds in \code{u_vec}.
}
}
\description{
\code{predict} method for class \code{"ithresh"}.  Predictive inferences can
either be based on a \emph{single training threshold} or using a weighted
average of inferences over \emph{multiple training thresholds}.  A single
threshold may chosen to be the best performing threshold, as judged by the
measure of predictive performance calculated by \code{\link{ithresh}} or
chosen by the user.  The weights used in the latter case are based on the
measures of predictive performance and prior probabilities assigned to the
training thresholds.  By default all thresholds are given the same
prior probability but the user can specify their own prior.
}
\details{
The function \code{\link[revdbayes]{predict.evpost}} is used to
  perform predictive based on the binomial-GP posterior sample generated
  using a given training threshold.  For mathematical details of the
  single threshold and multiple threshold cases see Sections 2.3 and 3 of
  Northrop et al. (2017) respectively.
}
\examples{
# Note:
#'  In the examples below validation thresholds rather higher than is
#   advisable have been used, with far fewer excesses than the minimum of
#   50 suggested by Jonathan and Ewans (2013).

# Gulf of Mexico significant wave heights, default priors.
u_vec_gom <- quantile(gom, probs = seq(0, 0.9, by = 0.05))
gom_cv <- ithresh(data = gom, u_vec = u_vec_gom, n_v = 3)

# Note: gom_cv$npy contains the correct value of npy (it was set in the
#       call to ithresh, via attr(gom, "npy").
#       If object$npy doesn't exist then the argument npy must be supplied
#       in the call to predict().

### Best training threshold based on the lowest validation threshold

# Predictive distribution function
best_p <- predict(gom_cv, n_years = c(100, 1000))
plot(best_p)

# Predictive density function
best_d <- predict(gom_cv, type = "d", n_years = c(100, 1000))
plot(best_d)

# Predictive intervals
best_i <- predict(gom_cv, n_years = c(100, 1000), type = "i", hpd = TRUE,
                  level = c(95, 99))
plot(best_i, which_int = "both")

# See which threshold was used
summary(gom_cv)

### All thresholds plus weighted average of inferences over all thresholds

# Predictive distribution function
all_p <- predict(gom_cv, which_u = "all")
plot(all_p)

# Predictive density function
all_d <- predict(gom_cv, which_u = "all", type = "d")
plot(all_d)
}
\references{
Northrop, P. J., Attalides, N. and Jonathan, P. (2017)
  Cross-validatory extreme value threshold selection and uncertainty
  with application to ocean storm severity.
  \emph{Journal of the Royal Statistical Society Series C: Applied
  Statistics}, \strong{66}(1), 93-120. \doi{10.1111/rssc.12159}
}
\seealso{
\code{\link{ithresh}} for threshold selection in the i.i.d. case
  based on leave-one-out cross-validation.

\code{\link{plot.ithreshpred}} for the S3 plot method for objects
  of class \code{ithreshpred}.
}
