% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/make-tk_make_timeseries.R
\name{tk_make_timeseries}
\alias{tk_make_timeseries}
\title{Intelligent date and date-time sequence creation}
\usage{
tk_make_timeseries(
  start_date,
  end_date,
  by,
  length_out = NULL,
  include_endpoints = TRUE,
  skip_values = NULL,
  insert_values = NULL
)
}
\arguments{
\item{start_date}{Used to define the starting date for date sequence generation.
Provide in "YYYY-MM-DD" format.}

\item{end_date}{Used to define the ending date for date sequence generation.
Provide in "YYYY-MM-DD" format.}

\item{by}{A character string, containing one of
\code{"sec"}, \code{"min"}, \code{"hour"}, \code{"day"}, \code{"week"}, \code{"month"}, \code{"quarter"} or \code{"year"}.
You can create regularly spaced sequences using phrases like \code{by = "10 min"}. See Details.}

\item{length_out}{Optional length of the sequence. Can be used instead of one of:
\code{start_date}, \code{end_date}, or \code{by}. Can be specified as a number or a time-based phrase.}

\item{include_endpoints}{Logical. Whether or not to keep the last value when \code{length_out} is
a time-based phrase. Default is \code{TRUE} (keep last value).}

\item{skip_values}{A sequence to skip}

\item{insert_values}{A sequence to insert}
}
\value{
A vector containing date or date-times
}
\description{
Improves on the \code{seq.Date()} and \code{seq.POSIXt()} functions by simplifying
into 1 function \code{tk_make_timeseries()}. Intelligently handles character dates
and logical assumptions based on user inputs.
}
\details{
The \code{tk_make_timeseries()} function handles both date and date-time sequences
automatically.
\itemize{
\item Parses date and date times from character
\item Intelligently guesses the sequence desired based on arguments provided
\item Handles spacing intelligently
\item When both \code{by} and \code{length_out} are missing, guesses either second or day sequences
\item Can skip and insert values if needed.
}

\strong{Start and End Date Specification}

Start and end dates can be specified in reduced time-based phrases:
\itemize{
\item \code{start_date = "2014"}: Is converted to "2014-01-01" (start of period)
\item \code{end_date = "2014"}: Is converted to "2014-12-31" (end of period)
\item \code{start_date = "2014-03"}: Is converted to "2014-03-01" (start of period)
\item \code{end_date = "2014-03"}: Is converted to "2014-03-31" (end of period)
}

A similar process can be used for date-times.

\strong{By: Daily Sequences}

Make a daily sequence with \code{tk_make_timeseries(by)}. Examples:
\itemize{
\item Every Day: \code{by = "day"}
\item Every 2-Weeks: \code{by = "2 weeks"}
\item Every 6-months: \code{by = "6 months"}
}

If missing, will guess \code{by = "day"}

\strong{By: Sub-Daily Sequences}

Make a sub-daily sequence with \code{tk_make_timeseries(by)}. Examples:
\itemize{
\item Every minute: \code{by = "min"}
\item Every 30-seconds: \code{by = "30 sec"}
\item Every 2-hours: \verb{by = "2 hours}
}

If missing, will guess \code{by = "sec"} if the start or end date is a date-time specification.

\strong{Length Out}

The \code{length_out} can be specified by number of observation or complex time-based expressions.
The following examples are all possible.
\itemize{
\item \code{length_out = 12} Creates 12 evenly spaced observations.
\item \code{length_out = "12 months"} Adjusts the end date so it falls on the 12th month.
}

\strong{Include Endpoint}

Sometimes the last date is not desired.
For example, if the user specifies \verb{length_out = 12 months}, the user may want the last value
to be the 12th month and not the 13th. Just toggle, \code{include_endpoint = FALSE} to obtain this
behavior.

\strong{Skip / Insert Timestamps}

Skips and inserts are performed after the sequence is generated. This means that if you use
the \code{length_out} parameter, the length may differ than the \code{length_out}.
}
\examples{
library(dplyr)
library(tidyquant)
library(timetk)

options(max.print = 50)

# ---- DATE ----

# Start + End, Guesses by = "day"
tk_make_timeseries("2017-01-01", "2017-12-31")

# Just Start
tk_make_timeseries("2017") # Same result

# Only dates in February, 2017
tk_make_timeseries("2017-02")

# Start + Length Out, Guesses by = "day"
tk_make_timeseries("2012", length_out = 6) # Guesses by = "day"

# Start + By + Length Out, Spacing 6 observations by monthly interval
tk_make_timeseries("2012", by = "1 month", length_out = 6)

# Start + By + Length Out, Phrase "1 year 6 months"
tk_make_timeseries("2012", by = "1 month",
                   length_out = "1 year 6 months", include_endpoints = FALSE)

# Going in Reverse, End + Length Out
tk_make_timeseries(end_date = "2012-01-01", by = "1 month",
                   length_out = "1 year 6 months", include_endpoints = FALSE)

# ---- DATE-TIME ----

# Start + End, Guesses by second
tk_make_timeseries("2016-01-01 01:01:02", "2016-01-01 01:01:04")

# Date-Time Sequence - By 10 Minutes
# - Converts to date-time automatically & applies 10-min interval
tk_make_timeseries("2017-01-01", "2017-01-02", by = "10 min")


# --- REMOVE / INCLUDE ENDPOINTS ----

# Last value in this case is desired
tk_make_timeseries("2017-01-01", by = "30 min", length_out = "6 hours")

# Last value in monthly case is not wanted
tk_make_timeseries("2012-01-01", by = "1 month",
                   length_out = "12 months",
                   include_endpoints = FALSE) # Removes unnecessary last value


# ---- SKIP & INSERT VALUES ----

tk_make_timeseries(
    "2011-01-01", length_out = 5,
    skip_values   = "2011-01-05",
    insert_values = "2011-01-06"
)

}
\seealso{
\itemize{
\item Intelligent date or date-time sequence creation: \code{\link[=tk_make_timeseries]{tk_make_timeseries()}}
\item Holidays and weekends: \code{\link[=tk_make_holiday_sequence]{tk_make_holiday_sequence()}}, \code{\link[=tk_make_weekend_sequence]{tk_make_weekend_sequence()}}, \code{\link[=tk_make_weekday_sequence]{tk_make_weekday_sequence()}}
\item Make future index from existing: \code{\link[=tk_make_future_timeseries]{tk_make_future_timeseries()}}
}
}
