% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GIRF.R, R/GIRF_plot.R
\name{GIRF}
\alias{GIRF}
\alias{GIRF.setar}
\alias{GIRF.linear}
\alias{GIRF.nlVar}
\alias{plot.GIRF_df}
\title{Generalized Impulse response Function (GIRF)}
\usage{
GIRF(object, n.ahead, seed = NULL, ...)

\method{GIRF}{setar}(
  object,
  n.ahead = 10,
  seed = NULL,
  n.hist = 20,
  n.shock = 20,
  R = 10,
  hist_li = NULL,
  shock_li = NULL,
  ...
)

\method{GIRF}{linear}(
  object,
  n.ahead = 10,
  seed = NULL,
  n.hist = 20,
  n.shock = 20,
  R = 10,
  hist_li = NULL,
  shock_li = NULL,
  ...
)

\method{GIRF}{nlVar}(
  object,
  n.ahead = 10,
  seed = NULL,
  n.hist = 20,
  n.shock = 20,
  R = 10,
  hist_li = NULL,
  shock_li = NULL,
  ...
)

\method{plot}{GIRF_df}(
  x,
  plot_type = c("density", "line"),
  n.ahead = c(1, 5, 10),
  var = unique(x$var)[1],
  n_simu = c(1, 2),
  ...
)
}
\arguments{
\item{object}{An object of class \code{\link{linear}}, \code{\link{setar}} or \code{nlVar} (\code{\link{TVAR}}, \code{\link{TVECM}})}

\item{n.ahead}{The number of steps ahead to compute}

\item{seed}{optional, the seed for the random numbers}

\item{n.hist}{The number of past histories to consider. Should be high, ideally size of data (minus lags).}

\item{n.shock}{The number of actual shocks to consider}

\item{R}{the number of draws to use for the n.ahead innovations}

\item{hist_li}{optional, a list of histories (each of same length as lags in the model). 
If not provided, \code{n.hist} histories will be randomly drawn for the original series.}

\item{shock_li}{optional, a list of innovations. 
If not provided, \code{n.shock} shocks will be randomly drawn from the estimated residuals.}

\item{x}{output of girf}

\item{plot_type}{plot: density (for each \code{n.ahead}), or line (for multipe \code{n_simu})?}

\item{var}{plot: which variable to plot?}

\item{n_simu}{line plot: which simulation to plot?}

\item{\ldots}{Further arguments passed to specific methods.}
}
\value{
A data-frame, with:
\describe{ 
\item{n_simu:}{Id for the simulation (total number is n.hist times n.shock)} 
\item{hist, shock}{History and shock used in the nth simulation}
\item{n.ahead:}{The forecasting horizon. Note the shocks happens at time 0}
\item{var:}{The variable (on which the shock happens, corresponds hence to the \code{response} argument in \code{irf})}
\item{sim_1, sim_2}{The average (over R times) simulation with the specific shock (sim_1) or with random shocks (sim_2). }
\item{girf}{The difference between sim_1 and sim_2}
}
}
\description{
Generates a GIRF for multiple innovations and histories
}
\details{
In a nonlinear model, the Impulse response Function (IRF) is not time-, scale- and sign-invariant as in linear models. 
To address this, Koop et al (1996) introduced the Generalized Impulse response Function (GIRF):
\deqn{GIRF(h,\delta,\omega_{t-1})=E[y_{t+h}|\epsilon_{t}=\delta,\epsilon_{t+h}=0,\omega_{t-1}]-E[y_{t+h}|\epsilon_{t}=0,\epsilon_{t+h}=0,\omega_{t-1}]}

It is the difference between two conditional expectations, one containing the shock of interest, the second one averaging it out. The averaging-out is done
by comparing against random innovations (unlike the IRF, which compares against innovation 0), 
The parameter \code{R} corresponds to the number of times this is done. 

The GIRF as defined here depends on the particular shock, as well as on the history. 
Koop et al (1996) suggest to draw multiple combinations of histories and innovations. This is done with arguments \code{n.hist} and \code{n.shock} 
(or, alternatively, provide one of, or both, \code{hist_li} and \code{hist_li} as list of histories and shocks).

The output is a data-frame containing the two average paths for each combinations of shocks and histories.
}
\examples{

## simulate a SETAR for the example. Higher regime more persistent (AR coef 0.5 instead of 0.2)
set <- setar.sim(B = c(0.5, 0.2, 0.5, 0.5), lag = 1, Thresh = 0.5, n = 500)
set_estim <- setar(set, m = 1)

## regime-specific IRF:
plot(irf(set_estim, regime = "L", boot = FALSE))
plot(irf(set_estim, regime = "H", boot = FALSE))

## GIRF
girf_out <- GIRF(set_estim, n.hist = 10, n.shock = 10) # smaller number for example only

## the GIRF shows a very fast convergence (the shock at n.ahead = 4 is already very close to 0)
plot(girf_out, n.ahead = 1:4)
## investigate a few specific GIRFS:
plot(girf_out, plot_type = "line", n_simu  = 1:5)
}
\references{
Koop, G, Pesaran, M. H. & Potter, S. M. (1996) Impulse response analysis in nonlinear multivariate models. Journal of Econometrics, 74, 119-147
}
\seealso{
\code{\link{irf.nlVar}} for the IRF, for linear models, or in case of non-linear models, for each regime.
}
\author{
Matthieu Stigler
}
