#' Time-Varying Coefficient Arrays of the Orthogonalised MA Represention
#'
#' Returns the estimated orthogonalised time-varying coefficient arrays of the
#' moving average representation of a stable \code{tvvar} object obtained with function \code{tvVAR}.
#' @param x An object of class \code{tvvar}, generated by tvVAR().
#' @param nstep	An integer specifying the number of othogonalised moving error coefficient
#' matrices to be calculated for each time t.
#' @param ortho.cov A character either 'const' if the error cov matrix must be estimated by a constant
#' or 'tv' if it is estimated as a time-varying matrix. Default is 'const'.
#' @param bw.cov A scalar (optional) with the bandwidth to estimate the errors variance-covariance matrix.
#' @return A list with an array of dimensions (obs x neq x neq nstep + 1) holding the estimated
#' time varying coefficients of the moving average representation, and the bandwidth used to estimate
#' the covariance matrix (optional).
#' @param ... Other parameters passed to specific methods.
#' @rdname tvPsi
#' @export
tvPsi<- function (x, nstep = 10, ortho.cov = "const", bw.cov = NULL, ...)
{
  if (!(class(x) == "tvvar"))
    stop("\nPlease provide an object of class 'tvvar', generated by tvVAR().\n")
  obs <- x$obs
  nstep <- abs(as.integer(nstep))
  neq <- x$neq
  X <- x$x
  params <- NCOL(X)
  Phi <- tvPhi(x, nstep = nstep)
  horizon <- dim(Phi)[4]
  Psi <- array(0, dim = dim(Phi))
  Sigma.hat <- array(0, dim=c(neq, neq, obs))
  if(ortho.cov == "tv")
  {
    tkernel <- x$tkernel
    est <- x$est
    if(is.null(bw.cov))
      bw.cov <- bwCov(x$residuals, tkernel = tkernel, est = est)
    Sigma.hat <- tvCov(x$residuals, bw = bw.cov, tkernel = tkernel, est = est)
  }
  for (t in 1:obs)
  {
    if(ortho.cov == "const")
      Sigma.hat[, , t] <- crossprod(x$residuals)/(obs - params)
    for (i in 1:horizon)
    {
      P <- try(chol(Sigma.hat[, , t]), silent = T)
      if (class(P) != "matrix")
      {
        Sigma.hat[, , t] <- as.matrix(nearPD(Sigma.hat[, , t])$mat)
        P <- try(chol(Sigma.hat[, , t]), silent = T)
      }
      Psi[t, ,  , i] <- Phi[t, , , i] %*% t(P)
    }
  }
  return(list(Psi = Psi, bw.cov = bw.cov))
}
