#' Retrieve Fiscal Data API in a single call
#'
#' @description
#'
#' `ustfd_simple()` aggregates the workflow for retrieving data from the API
#' into a single call.
#'
#' @param endpoint required string representing an API endpoint
#' @param filter optional list used to subset the data. known filter operators
#' are '>', '>=', '<', '<=', '=', and 'in'
#' @param fields optional string vector of the fields to be retrieved
#' @param sort optional string or string vector. Ordering defaults to ascending,
#' to specify descending order precede the field name with '-'
#' @param page_size optional integer for pagination
#' @param page_number optional integer for pagination
#' @param user_agent optional string
#'
#' @return a list containing the following items
#'  * `meta` - the metadata returned by the API
#'  * `data` - the payload returned by the API in table form.
#'  See [`ustfd_response_payload()`]
#'
#' @export
#'
#' @family ustfd_user
#'
#' @examples
#' \dontrun{
#' library(ustfd)
#'
#' exchange_rates <- ustfd_simple(
#'   '/v1/accounting/od/rates_of_exchange',
#'    fields = c(
#'     'country_currency_desc', 'exchange_rate','record_date','effective_date'
#'    ),
#'    filter = list(
#'      record_date = c('>=' = '2020-01-01'),
#'      country_currency_desc = list('in' = c('Canada-Dollar','Mexico-Peso'))
#'    )
#' )
#' }

ustfd_simple <- function(
  endpoint, filter=NA, fields=NA, sort=NA, page_size=NA, page_number=NA,
  user_agent='http://github.com/groditi/ustfd'
  ){
  query <- ustfd_query(endpoint, filter, fields, sort, page_size, page_number)
  response <- ustfd_request(query, user_agent)
  return(
    list(
      meta = ustfd_response_meta_object(response),
      data = ustfd_response_payload(response)
    )
  )

}

#' Retrieve Data From the U.S. Bureau Of the Fiscal Service API
#'
#' @description
#'
#' `ustfd_request()`  will execute queries against the Fiscal Data API. Queries
#' can generated using [ustfd_query()].
#'
#' @param query list generated by one of the query generating functions
#' @param user_agent string, optional
#' @param process_response function, optional. processes the `httr` response
#'  object. Defaults to [`ustfd_json_response()`] which will return the JSON
#'  payload parsed into a list
#' @param ... further arguments will be passed to `process_response` when called
#'
#' @return a httr response object
#'
#' @export
#'
#' @family ustfd_low_level
#'
#' @examples
#' \dontrun{
#' library(ustfd)
#' query <- ustfd_query('/v1/accounting/dts/dts_table_4', sort =c('-record_date'))
#' response <- ustfd_request(query)
#' payload_table <- ustfd_response_payload(response)
#' payload_meta <- ustfd_response_meta_object(response)
#' }

ustfd_request <- function(
  query,
  user_agent='http://github.com/groditi/ustfd',
  process_response = ustfd_json_response,
  ...
){
  response <- httr::GET(utils::URLdecode(ustfd_url(query)), httr::user_agent(user_agent))
  httr::stop_for_status(response)
  if(response$status_code > 200)
    stop(httr::http_status(response)$message)

  return(process_response(response, ...))
}

#' Process JSON Response of a Successful API Query
#'
#' @description
#'
#' `ustfd_json_response()`  will process the response to a successful request
#' from Fiscal Data API and translate a JSON object into a R data structure.
#'
#' @param response an httr response returned by [ustfd_request()]
#' @param ... additional arguments passed to `httr::content`
#'
#' @return a list
#'
#' @export
#'
#' @family ustfd_low_level
#'
#' @examples
#' \dontrun{
#' library(ustfd)
#' query <- ustfd_query('/v1/accounting/dts/dts_table_4', sort =c('-record_date'))
#' response <- ustfd_request(query)
#' payload_table <- ustfd_response_payload(response)
#' payload_meta <- ustfd_response_meta_object(response)
#' }

ustfd_json_response <- function(response, ...){
  if(httr::headers(response)[['content-type']] != 'application/json')
    stop(paste(httr::headers(response)[['content-type']], 'is not JSON'))

  parsed <- httr::content(response, as = 'parsed', simplifyVector = FALSE, ...)
  if('error' %in% names(parsed))
    stop(parsed$message)

  return(parsed)
}

#' Extract Metadata From Parsed API Response
#'
#' @description
#'
#' `ustfd_response_meta_object()`  will return the meta object included in a
#' successful API response. The meta object is a list with the following items:
#'
#' * `count` - the number of records in the response
#' * `labels` - a named list of labels for each field
#' * `dataTypes` - a named list describing the data type for each field
#' * `dataFormats` - a named list describing the data format for each field
#' * `total-count` - the total number of records matching the query
#' * `total-pages` - the total number of pages of records matching the query
#'
#'
#' @param response a parsed response returned by [ustfd_json_response()]
#'
#' @return a list
#'
#' @export
#'
#' @family ustfd_low_level
#'
#' @examples
#' \dontrun{
#' library(ustfd)
#' query <- ustfd_query('/v1/accounting/dts/dts_table_4', sort =c('-record_date'))
#' response <- ustfd_request(query)
#' payload_table <- ustfd_response_payload(response)
#' payload_meta <- ustfd_response_meta_object(response)
#' }

ustfd_response_meta_object <- function(response){
  response$meta
}

#' Extract Payload as Table From Parsed API Response
#'
#' @description
#'
#' `ustfd_response_payload()` will return the results of the query in tabular
#' format in the form of a tibble with one column for each field returned and
#' one row for every record returned in the same order they were returned.
#'
#' @param response a parsed response returned by [ustfd_json_response()]
#'
#' @return a tibble
#'
#' @export
#'
#' @family ustfd_low_level
#'
#' @examples
#' \dontrun{
#' library(ustfd)
#' query <- ustfd_query('/v1/accounting/dts/dts_table_4', sort =c('-record_date'))
#' response <- ustfd_request(query)
#' payload_table <- ustfd_response_payload(response)
#' payload_meta <- ustfd_response_meta_object(response)
#' }
#'
ustfd_response_payload <- function(response){

  payload <- dplyr::mutate(
    dplyr::bind_rows(response$data),
    dplyr::across(dplyr::everything(), ~ifelse(.x == 'null', NA, .x))
  )

  #find columns where all values are null
  null_tests <- purrr::map_lgl(names(payload), ~any(!is.na(payload[[.x]])))
  names(null_tests) <- names(payload)
  all_nulls <- names(null_tests[!null_tests])

  date_types <- c('DATE')
  percent_types <- c('PERCENTAGE')
  currency_types <- c('CURRENCY')
  numeric_types <- c('YEAR','DAY','MONTH','QUARTER','NUMBER')


  meta_types <- ustfd_response_meta_object(response)$dataTypes
  #exclude all-null columns from typecasting based on metadata
  meta_types <- meta_types[!names(meta_types) %in% all_nulls]

  numeric_cols <- names(meta_types[meta_types %in% numeric_types])
  date_cols <- names(meta_types[meta_types %in% date_types])
  percent_cols <- names(meta_types[meta_types %in% percent_types])
  currency_cols <- names( meta_types[meta_types %in% currency_types])

  if(length(numeric_cols) >= 1)
    payload <- dplyr::mutate(
      payload, dplyr::across(dplyr::all_of(numeric_cols), as.numeric)
    )

  if(length(date_cols) >= 1)
    payload <- dplyr::mutate(
      payload, dplyr::across(dplyr::all_of(date_cols), lubridate::ymd)
    )

  if(length(currency_cols) >= 1)
    payload <- dplyr::mutate(
      payload,
      dplyr::across(dplyr::all_of(currency_cols), readr::parse_number)
    )

  if(length(percent_cols) >= 1)
    payload <- dplyr::mutate(
      payload,
      dplyr::across(dplyr::all_of(percent_cols), ~readr::parse_number(.x)*.01)
    )

  return(payload)
}


