% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/walker.R
\name{walker}
\alias{walker}
\title{Bayesian regression with random walk coefficients}
\usage{
walker(formula, data, sigma_y_prior, beta_prior, init, chains,
  return_x_reg = FALSE, ...)
}
\arguments{
\item{formula}{An object of class \code{\link[stats]{formula}} with additional terms 
\code{rw1} and/or \code{rw2} e.g. \code{y ~ x1 + rw1(~ -1 + x2)}. See details.}

\item{data}{An optional data.frame or object coercible to such, as in \code{\link[stats]{lm}}.}

\item{sigma_y_prior}{A vector of length two, defining the truncated Gaussian prior for 
the observation level standard deviation. Not used in \code{walker_glm}.}

\item{beta_prior}{A length vector of length two which defines the 
prior mean and standard deviation of the Gaussian prior for time-invariant coefficients}

\item{init}{Initial value specification, see \code{\link[rstan]{sampling}}. 
Note that compared to default in \code{rstan}, here the default is a to sample from the priors.}

\item{chains}{Number of Markov chains. Default is 4.}

\item{return_x_reg}{If \code{TRUE}, does not perform sampling, but instead returns the matrix of 
predictors after processing the \code{formula}.}

\item{...}{Further arguments to \code{\link[rstan]{sampling}}.}
}
\value{
A list containing the \code{stanfit} object, observations \code{y},
  and covariates \code{xreg} and \code{xreg_new}.
}
\description{
Function \code{walker} performs Bayesian inference of a linear 
regression model with time-varying, random walk regression coefficients, 
i.e. ordinary regression model where instead of constant coefficients the 
coefficients follow first or second order random walks. 
All Markov chain Monte Carlo computations are done using Hamiltonian 
Monte Carlo provided by Stan, using a state space representation of the model 
in order to marginalise over the coefficients for efficient sampling.
}
\details{
The \code{rw1} and \code{rw2} functions used in the formula define new formulas 
for the first and second order random walks. In addition, these functions 
need to be supplied with priors for initial coefficients and the 
standard devitions. For second order random walk model, these sigma priors 
correspond to the standard deviation of slope distrurbances. For \code{rw2}, 
also a prior for the initial slope needs to be defined. See examples.
}
\note{
Beware of overfitting and identifiability issues. In particular, 
be careful in not defining multiple intercept terms (only one should be present).
}
\examples{

## note very low number of iterations for the CRAN checks

rw1_fit <- walker(Nile ~ -1 + 
  rw1(~ 1, 
    beta_prior = c(1000, 100), 
    sigma_prior = c(0, 100)), 
  sigma_y_prior = c(0, 100), 
  iter = 50, chains = 1)
  
rw2_fit <- walker(Nile ~ -1 + 
  rw2(~ 1,
    beta_prior = c(1000, 100), 
    sigma_prior = c(0, 100), 
    slope_prior = c(0, 100)), 
  sigma_y_prior = c(0, 100), 
  iter = 50, chains = 1)
  
g_y <- geom_point(data = data.frame(y = Nile, x = time(Nile)), 
  aes(x, y, alpha = 0.5), inherit.aes = FALSE) 
g_rw1 <- plot_coefs(rw1_fit) + g_y
g_rw2 <- plot_coefs(rw2_fit) + g_y
if(require("gridExtra")) {
  grid.arrange(g_rw1, g_rw2, ncol=2, top = "RW1 (left) versus RW2 (right)")
} else {
  g_rw1
  g_rw2
}
\dontrun{
y <- window(log10(UKgas), end = time(UKgas)[100])
n <- 100
cos_t <- cos(2 * pi * 1:n / 4)
sin_t <- sin(2 * pi * 1:n / 4)
dat <- data.frame(y, cos_t, sin_t)
fit <- walker(y ~ -1 + 
  rw1(~ cos_t + sin_t, beta_prior = c(0, 10), sigma_prior = c(0, 2)), 
  sigma_y_prior = c(0, 10), data = dat, chains = 1, iter = 250)
print(fit$stanfit, pars = c("sigma_y", "sigma_rw1"))

plot_coefs(fit)
# posterior predictive check:
pp_check(fit)

newdata <- data.frame(
  cos_t = cos(2 * pi * 101:108 / 4), 
  sin_t = sin(2 * pi * 101:108 / 4))
pred <- predict(fit, newdata)
plot_predict(pred)
}

}
\seealso{
\code{\link{walker_glm}} for non-Gaussian models.
}
