% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cwt_wst.R
\name{cwt_wst}
\alias{cwt_wst}
\title{Continuous wavelet transform}
\usage{
cwt_wst(signal,
               dt = 1,
               scales = NULL,
               powerscales = TRUE,
               wname = c("MORLET", "DOG", "PAUL", "HAAR", "HAAR2"),
               wparam = NULL,
               waverad = NULL,
               border_effects = c("BE", "PER", "SYM"),
               makefigure = TRUE,
               time_values = NULL,
               energy_density = FALSE,
               figureperiod = TRUE,
               xlab = "Time",
               ylab = NULL,
               main = NULL,
               zlim = NULL)
}
\arguments{
\item{signal}{A vector containing the signal whose wavelet transform is wanted.}

\item{dt}{Numeric. The time step of the signal.}

\item{scales}{A vector containing the wavelet scales at wich the CWT
is computed. This can be either a vector with all the scales or, following Torrence
and Compo 1998, a vector of 3 elements with the minimum scale, the maximum scale and
the number of suboctaves per octave (in this case, \code{powerscales} must be TRUE in
order to construct power 2 scales using a base 2 logarithmic scale). If \code{scales}
is NULL, they are automatically constructed.}

\item{powerscales}{Logical. It must be TRUE (default) in these cases:
\itemize{
\item If \code{scales} are power 2 scales, i.e. they use a base 2 logarithmic scale.
\item If we want to construct power 2 scales automatically. In this case, \code{scales}
must be \code{NULL}.
\item If we want to construct power 2 scales from \code{scales}. In this case,
\code{length(scales)} must be 3.
}}

\item{wname}{A string, equal to "MORLET", "DOG", "PAUL", "HAAR" or "HAAR2". The
difference between "HAAR" and "HAAR2" is that "HAAR2" is more accurate but slower.}

\item{wparam}{The corresponding nondimensional parameter for the wavelet function
(Morlet, DoG or Paul).}

\item{waverad}{Numeric. The radius of the wavelet used in the computations for the cone
of influence. If it is not specified, it is asumed to be \eqn{\sqrt{2}} for Morlet and DoG,
\eqn{1/\sqrt{2}} for Paul and 0.5 for Haar.}

\item{border_effects}{String, equal to "BE", "PER" or "SYM", which indicates how to
manage the border effects which arise usually when a convolution is performed on
finite-lenght signals.
\itemize{
\item "BE": Padding time series with zeroes.
\item "PER": Using boundary wavelets (periodization of the original time series).
\item "SYM": Using a symmetric catenation of the original time series.
}}

\item{makefigure}{Logical. If TRUE (default), a figure with the wavelet power spectrum
is plotted.}

\item{time_values}{A numerical vector of length \code{length(signal)} containing custom
time values for the figure. If NULL (default), it will be computed starting at 0.}

\item{energy_density}{Logical. If TRUE (default), divide the wavelet power spectrum by
the scales in the figure and so, values for different scales are comparable.}

\item{figureperiod}{Logical. If TRUE (default), periods are used in the figure instead
of scales.}

\item{xlab}{A string giving a custom X axis label.}

\item{ylab}{A string giving a custom Y axis label. If NULL (default) the Y label is
either "Scale" or "Period" depending on the value of \code{figureperiod}.}

\item{main}{A string giving a custom main title for the figure. If NULL
(default) the main title is either "Wavelet Power Spectrum / Scales" or "Wavelet Power
Spectrum" depending on the value of \code{energy_density}.}

\item{zlim}{A vector of length 2 with the limits for the z-axis (the color bar).}
}
\value{
A list with the following fields:
\itemize{
\item \code{coefs}: A matrix of size \code{length(signal)} x \code{length(scales)},
containing the CWT coefficients of the signal.
\item \code{scales}: The vector of scales.
\item \code{fourierfactor}: A factor for converting scales into periods.
\item \code{coi_maxscale}: A vector of length \code{length(signal)} containing the
values of the maximum scale from which there are border effects at each time.
}
}
\description{
This function computes the continuous wavelet transform for some families of wavelet
bases: "MORLET", "DOG", "PAUL" and "HAAR".
It is a translation from the Matlab(R) function published by Torrence and Compo
(Torrence & Compo, 1998).

The difference between \code{cwt_wst} and \code{cwt} from package \code{Rwave} is that
\code{cwt_wst} normalizes using \eqn{L^2} and \code{cwt} uses \eqn{L^1}.
}
\section{References}{


C. Torrence, G. P. Compo. A practical guide to wavelet analysis. B. Am. Meteorol. Soc.
79 (1998), 61–78.
}

\examples{
dt <- 0.1
time <- seq(0, 50, dt)
signal <- c(sin(pi * time), sin(pi * time / 2))
cwt <- cwt_wst(signal = signal, dt = dt, energy_density = TRUE)

}
