% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AFparfrailty.R
\name{AFparfrailty}
\alias{AFparfrailty}
\title{Attributable fraction function based on a Weibull gamma-frailty model as a \code{\link[stdReg]{parfrailty}} object (commonly used for cohort sampling family designs with time-to-event outcomes).}
\usage{
AFparfrailty(object, data, exposure, times, clusterid)
}
\arguments{
\item{object}{a fitted Weibull gamma-parfrailty object of class "\code{\link[stdReg]{parfrailty}}".}

\item{data}{an optional data frame, list or environment (or object coercible by \code{as.data.frame} to a data frame) containing the variables in the model. If not found in data, the variables are taken from environment (\code{formula}), typically the environment from which the function is called.}

\item{exposure}{the name of the exposure variable as a string. The exposure must be binary (0/1) where unexposed is coded as 0.}

\item{times}{a scalar or vector of time points specified by the user for which the attributable fraction function is estimated. If not specified the observed death times will be used.}

\item{clusterid}{the name of the cluster identifier variable as a string, if data are clustered.}
}
\value{
\item{AF.est}{estimated attributable fraction function for every time point specified by \code{times}.}

\item{AF.var}{estimated variance of \code{AF.est}. The variance is obtained by combining the delta methods with the sandwich formula.}

\item{S.est}{estimated factual survival function; \eqn{S(t)}.}

\item{S.var}{estimated variance of \code{S.est}. The variance is obtained by the sandwich formula.}

\item{S0.est}{estimated counterfactual survival function if exposure would be eliminated; \eqn{S_0(t)}{S0(t)}.}

\item{S0.var}{estimated variance of \code{S0.est}. The variance is obtained by the sandwich formula.}
}
\description{
\code{AFparfrailty} estimates the model-based adjusted attributable fraction function from a shared Weibull gamma-frailty model in form of a  \code{\link[stdReg]{parfrailty}} object. This model is commonly used for data from cohort sampling familty designs with time-to-event outcomes.
}
\details{
\code{AFparfrailty} estimates the attributable fraction for a time-to-event outcome
under the hypothetical scenario where a binary exposure \code{X} is eliminated from the population.
The estimate is adjusted for confounders \code{Z} by the shared frailty model (\code{\link[stdReg]{parfrailty}}).
The baseline hazard is assumed to follow a Weibull distribution and the unobserved shared frailty effects \code{U} are assumed to be gamma distributed.
Let the AF function be defined as
\deqn{AF=1-\frac{\{1-S_0(t)\}}{\{1-S(t)\}}}{AF = 1 - {1 - S0(t)} / {1 - S(t)}}
where \eqn{S_0(t)}{S0(t)} denotes the counterfactual survival function for the event if
the exposure would have been eliminated from the population at baseline and \eqn{S(t)} denotes the factual survival function.
If \code{Z} and \code{U} are sufficient for confounding control, then \eqn{S_0(t)}{S0(t)} can be expressed as \eqn{E_Z\{S(t\mid{X=0,Z })\}}{E_z{S(t|X=0,Z)}}.
The function uses a fitted Weibull gamma-frailty model to estimate \eqn{S(t\mid{X=0,Z})}{S(t|X=0,Z)}, and the marginal sample distribution of \code{Z}
to approximate the outer expectation. A clustered sandwich formula is used in all variance calculations.
}
\examples{
# Example 1: clustered data with frailty U
expit <- function(x) 1 / (1 + exp( - x))
n <- 100
m <- 2
alpha <- 1.5
eta <- 1
phi <- 0.5
beta <- 1
id <- rep(1:n,each=m)
U <- rep(rgamma(n, shape = 1 / phi, scale = phi), each = m)
Z <- rnorm(n * m)
X <- rbinom(n * m, size = 1, prob = expit(Z))
# Reparametrize scale as in rweibull function
weibull.scale <- alpha / (U * exp(beta * X)) ^ (1 / eta)
t <- rweibull(n * m, shape = eta, scale = weibull.scale)

# Right censoring
c <- runif(n * m, 0, 10)
delta <- as.numeric(t < c)
t <- pmin(t, c)

data <- data.frame(t, delta, X, Z, id)

# Fit a parfrailty object
library(stdReg)
fit <- parfrailty(formula = Surv(t, delta) ~ X + Z + X * Z, data = data, clusterid = "id")
summary(fit)

# Estimate the attributable fraction from the fitted frailty model

time <- c(seq(from = 0.2, to = 1, by = 0.2))

AFparfrailty_est <- AFparfrailty(object = fit, data = data, exposure = "X",
                                  times = time, clusterid = "id")
summary(AFparfrailty_est)
plot(AFparfrailty_est, CI = TRUE, ylim=c(0.1,0.7))
}
\seealso{
\code{\link[stdReg]{parfrailty}} used for fitting the Weibull gamma-frailty and \code{\link[stdReg]{stdParfrailty}} used for standardization of a \code{parfrailty} object.
}
\author{
Elisabeth Dahlqwist, Arvid \enc{Sjölander}{Sjolander}
}
