% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/portion.R
\name{portion}
\alias{portion}
\alias{portion_R}
\alias{portion_IR}
\alias{portion_I}
\alias{portion_SI}
\alias{portion_S}
\alias{portion_df}
\title{Calculate resistance of isolates}
\source{
\strong{M39 Analysis and Presentation of Cumulative Antimicrobial Susceptibility Test Data, 4th Edition}, 2014, \emph{Clinical and Laboratory Standards Institute (CLSI)}. \url{https://clsi.org/standards/products/microbiology/documents/m39/}.

Wickham H. \strong{Tidy Data.} The Journal of Statistical Software, vol. 59, 2014. \url{http://vita.had.co.nz/papers/tidy-data.html}
}
\usage{
portion_R(..., minimum = 30, as_percent = FALSE,
  also_single_tested = FALSE)

portion_IR(..., minimum = 30, as_percent = FALSE,
  also_single_tested = FALSE)

portion_I(..., minimum = 30, as_percent = FALSE,
  also_single_tested = FALSE)

portion_SI(..., minimum = 30, as_percent = FALSE,
  also_single_tested = FALSE)

portion_S(..., minimum = 30, as_percent = FALSE,
  also_single_tested = FALSE)

portion_df(data, translate_ab = getOption("get_antibiotic_names",
  "official"), minimum = 30, as_percent = FALSE, combine_IR = FALSE)
}
\arguments{
\item{...}{one or more vectors (or columns) with antibiotic interpretations. They will be transformed internally with \code{\link{as.rsi}} if needed. Use multiple columns to calculate (the lack of) co-resistance: the probability where one of two drugs have a resistant or susceptible result. See Examples.}

\item{minimum}{the minimum allowed number of available (tested) isolates. Any isolate count lower than \code{minimum} will return \code{NA} with a warning. The default number of \code{30} isolates is advised by the Clinical and Laboratory Standards Institute (CLSI) as best practice, see Source.}

\item{as_percent}{a logical to indicate whether the output must be returned as a hundred fold with \% sign (a character). A value of \code{0.123456} will then be returned as \code{"12.3\%"}.}

\item{also_single_tested}{a logical to indicate whether (in combination therapies) also observations should be included where not all antibiotics were tested, but at least one of the tested antibiotics contains a target interpretation (e.g. S in case of \code{portion_S} and R in case of \code{portion_R}). \strong{This would lead to selection bias in almost all cases.}}

\item{data}{a \code{data.frame} containing columns with class \code{rsi} (see \code{\link{as.rsi}})}

\item{translate_ab}{a column name of the \code{\link{antibiotics}} data set to translate the antibiotic abbreviations to, using \code{\link{abname}}. This can be set with \code{\link{getOption}("get_antibiotic_names")}.}

\item{combine_IR}{a logical to indicate whether all values of I and R must be merged into one, so the output only consists of S vs. IR (susceptible vs. non-susceptible)}
}
\value{
Double or, when \code{as_percent = TRUE}, a character.
}
\description{
These functions can be used to calculate the (co-)resistance of microbial isolates (i.e. percentage of S, SI, I, IR or R). All functions support quasiquotation with pipes, can be used in \code{dplyr}s \code{\link[dplyr]{summarise}} and support grouped variables, see \emph{Examples}.

\code{portion_R} and \code{portion_IR} can be used to calculate resistance, \code{portion_S} and \code{portion_SI} can be used to calculate susceptibility.\cr
}
\details{
\strong{Remember that you should filter your table to let it contain only first isolates!} Use \code{\link{first_isolate}} to determine them in your data set.

These functions are not meant to count isolates, but to calculate the portion of resistance/susceptibility. Use the \code{\link[AMR]{count}} functions to count isolates. \emph{Low counts can infuence the outcome - these \code{portion} functions may camouflage this, since they only return the portion albeit being dependent on the \code{minimum} parameter.}

\code{portion_df} takes any variable from \code{data} that has an \code{"rsi"} class (created with \code{\link{as.rsi}}) and calculates the portions R, I and S. The resulting \emph{tidy data} (see Source) \code{data.frame} will have three rows (S/I/R) and a column for each variable with class \code{"rsi"}.

The old \code{\link{rsi}} function is still available for backwards compatibility but is deprecated.
\if{html}{
  \cr\cr
  To calculate the probability (\emph{p}) of susceptibility of one antibiotic, we use this formula:
  \out{<div style="text-align: center;">}\figure{combi_therapy_2.png}\out{</div>}
  To calculate the probability (\emph{p}) of susceptibility of more antibiotics (i.e. combination therapy), we need to check whether one of them has a susceptible result (as numerator) and count all cases where all antibiotics were tested (as denominator). \cr
  \cr
  For two antibiotics:
  \out{<div style="text-align: center;">}\figure{combi_therapy_2.png}\out{</div>}
  \cr
  For three antibiotics:
  \out{<div style="text-align: center;">}\figure{combi_therapy_2.png}\out{</div>}
  \cr
  And so on.
}
}
\section{Read more on our website!}{

On our website \url{https://msberends.gitlab.io/AMR} you can find \href{https://msberends.gitlab.io/AMR/articles/AMR.html}{a comprehensive tutorial} about how to conduct AMR analysis, the \href{https://msberends.gitlab.io/AMR/reference}{complete documentation of all functions} (which reads a lot easier than here in R) and \href{https://msberends.gitlab.io/AMR/articles/WHONET.html}{an example analysis using WHONET data}.
}

\examples{
# septic_patients is a data set available in the AMR package. It is true, genuine data.
?septic_patients

# Calculate resistance
portion_R(septic_patients$amox)
portion_IR(septic_patients$amox)

# Or susceptibility
portion_S(septic_patients$amox)
portion_SI(septic_patients$amox)

# Do the above with pipes:
library(dplyr)
septic_patients \%>\% portion_R(amox)
septic_patients \%>\% portion_IR(amox)
septic_patients \%>\% portion_S(amox)
septic_patients \%>\% portion_SI(amox)

septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(p = portion_S(cipr),
            n = n_rsi(cipr)) # n_rsi works like n_distinct in dplyr

septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(R = portion_R(cipr, as_percent = TRUE),
            I = portion_I(cipr, as_percent = TRUE),
            S = portion_S(cipr, as_percent = TRUE),
            n1 = count_all(cipr), # the actual total; sum of all three
            n2 = n_rsi(cipr),     # same - analogous to n_distinct
            total = n())          # NOT the number of tested isolates!

# Calculate co-resistance between amoxicillin/clav acid and gentamicin,
# so we can see that combination therapy does a lot more than mono therapy:
septic_patients \%>\% portion_S(amcl)       # S = 67.1\%
septic_patients \%>\% count_all(amcl)       # n = 1576

septic_patients \%>\% portion_S(gent)       # S = 74.0\%
septic_patients \%>\% count_all(gent)       # n = 1855

septic_patients \%>\% portion_S(amcl, gent) # S = 92.0\%
septic_patients \%>\% count_all(amcl, gent) # n = 1517


septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(cipro_p = portion_S(cipr, as_percent = TRUE),
            cipro_n = count_all(cipr),
            genta_p = portion_S(gent, as_percent = TRUE),
            genta_n = count_all(gent),
            combination_p = portion_S(cipr, gent, as_percent = TRUE),
            combination_n = count_all(cipr, gent))

# Get portions S/I/R immediately of all rsi columns
septic_patients \%>\%
  select(amox, cipr) \%>\%
  portion_df(translate = FALSE)

# It also supports grouping variables
septic_patients \%>\%
  select(hospital_id, amox, cipr) \%>\%
  group_by(hospital_id) \%>\%
  portion_df(translate = FALSE)


\dontrun{

# calculate current empiric combination therapy of Helicobacter gastritis:
my_table \%>\%
  filter(first_isolate == TRUE,
         genus == "Helicobacter") \%>\%
  summarise(p = portion_S(amox, metr),  # amoxicillin with metronidazole
            n = count_all(amox, metr))
}
}
\seealso{
\code{\link[AMR]{count}_*} to count resistant and susceptible isolates.
}
\keyword{antibiotics}
\keyword{isolate}
\keyword{isolates}
\keyword{resistance}
\keyword{rsi}
\keyword{rsi_df}
\keyword{susceptibility}
