\name{amova.rda}
\alias{amova.rda}

\title{Analysis of Molecular Variance (AMOVA) via Redundancy Analysis}

\description{
The main objective of the function is to illustrate how information on 'mean squares' can be extracted from redundancy analysis results. For balanced data sets, the final 'Phi' statistics are expected to be the same as those obtained with \code{\link[poppr]{poppr.amova}}. The function can only handle 1-level (population) and 2-level (eg, region/population or population/subpopulation) hierarchies.
}

\usage{

amova.rda(x, x.data)

}

\arguments{

\item{x}{ Result of redundancy analysis fitted via \code{\link[vegan]{rda}}. }
\item{x.data}{ Data used to fit the redundancy analysis via \code{\link[BiodiversityR]{ensemble.VIF.dataframe}}. } 
  
}
\details{
A similar analysis of the relationship between AMOVA and RDA is given by Kindt (2020), also exploring relationships with the Multivariate Analysis of Variance Using Distance Matrices methods that are available via \code{\link[vegan]{adonis2}}. For discussions on AMOVA and its relationship with the matrix of squared Euclidean distances between individuals, see Michalakis and Excoffier (1996), Peakall et al. (1995) or Meirmans and Liu (2018) (the last article also shows how AMOVA can be expanded beyond haploid and diploid organisms). These authors provide the coefficients (equations 3, 4a and 4b) that need to be used with unbalanced numbers of individuals in populations and/or groups, which could then be used to obtain the exact AMOVA statistics from the estimated Mean Squares (there is no practical point in doing this as AMOVA is available via \code{\link[poppr]{poppr.amova}}).

I have also cross-checked the results with GenAlEx 6.5 (Peakall and Smouse 2012), using the export function of \code{\link[poppr]{genind2genalex}} to cross-check that RDA obtains the correct Sums-of-Squares and Mean-Squares. Important to note here is that the values in GenAlEx are obtained from only the upper part of the distance matrix, hence these are exactly 50 percent of the Sums-of-Squares from RDA.

Note that for diploid organisms, the option here is not to include the hierarchical level within individuals.
}

\value{
The function proceeds with Analysis of Molecular Variance from data generated by redundancy analysis (RDA).
}

\references{
Kindt R. 2020. Analysis of Molecular Variance (AMOVA) with vegan and BiodiversityR, including a graphical method to identify potential migrants. \url{https://rpubs.com/Roeland-KINDT}

Meirmans PG and Liu S. 2018. Analysis of Molecular Variance (AMOVA) for Autopolyploids. Front. Ecol. Evol., 23 May 2018. \doi{10.3389/fevo.2018.00066}

Michalakis Y and Excoffier L. 1996. A Generic Estimation of Population Subdivision Using Distances Between Alleles With Special Reference for Microsatellite Loci. Genetics 142: 1061-1064.

Peakall R, Smouse PE and Huff DR. 1995. Evolutionary implications of allozyme and RAPD variation in diploid populations of dioecious buffalograss (Buchloe dactyloides (Nutt.) Engelm.). \doi{10.1111/j.1365-294X.1995.tb00203.x}.

Peakall R and Smouse PE. 2012. GenAlEx 6.5: genetic analysis in Excel. Population genetic software for teaching and research an update. \doi{10.1093/bioinformatics/bts460}.

}

\author{Roeland Kindt (World Agroforestry, CIFOR-ICRAF)}

\examples{

\dontrun{

library(poppr)
library(BiodiversityR) # also loads vegan

# Example with 1 level
data(nancycats)
names(alleles(nancycats)) <- locNames(nancycats) # small bug in this data set
nancycats2 <- missingno(nancycats, type = "loci", cutoff=0)
nancy.dist <- vegdist(nancycats2@tab, method="euc")

# Same method as in GenAlEx via the squared Euclidean distance
nancy.dist.amova <- poppr.amova(nancycats2, ~Population, within=FALSE,
                                dist=nancy.dist, squared=FALSE)
nancy.dist.amova

# Via vegan::rda
library(vegan)
nancycats.rda.data <- data.frame(Population=nancycats2@pop)

nancycats.rda.result <- rda(nancycats2@tab ~ Population, data=nancycats.rda.data)  
nancycats.rda.result
amova.rda(nancycats.rda.result, nancycats.rda.data) 

# Example with 2 levels
# Same example as for poppr::poppr.amova
data(Aeut)
strata(Aeut) <- other(Aeut)$population_hierarchy[-1]
agc <- as.genclone(Aeut)
agc
amova.result <- poppr.amova(agc, ~Pop/Subpop, within=FALSE)
amova.result

agc.rda.data <- data.frame(as.matrix(agc$other$population_hierarchy))
agc.rda.data[, 1] <- as.factor(agc.rda.data[, 1])
agc.rda.data[, 2] <- as.factor(agc.rda.data[, 2])
agc.rda.data[, 3] <- as.factor(agc.rda.data[, 3])
str(agc.rda.data)

agc.rda.result <- rda(agc@tab ~ Pop + Pop_Subpop, data=agc.rda.data)  
agc.rda.result
amova.rda(agc.rda.result, agc.rda.data) 

# Example with a balanced data set
# library(BiodiversityR)
data(warcom)
data(warenv)

warburgia.genind <- genind(warcom)
warburgia.genind@strata <- warenv
poppr.amova(warburgia.genind, ~ population, within=FALSE)

warburgia.rda <- rda(warcom ~ population, data=warenv)
warburgia.rda
amova.rda(warburgia.rda, warenv)

}
}




