\name{BANOVA.mediation}
\alias{BANOVA.mediation}
\alias{print.BANOVA.mediation}
\title{Mediation analysis based on BANOVA models}
\description{
\code{BANOVA.mediation} conducts mediation and moderated mediation analysis based on various BANOVA models with a single mediator and a casual variable. 
}
\usage{
BANOVA.mediation(sol_1, sol_2, xvar, mediator, individual = F, 
return_posterior_samples = F,  multi_samples_beta1_raw_m = NULL)
\method{print}{BANOVA.mediation}(x, ...)
}
\arguments{
  \item{sol_1}{an  object  of  class  \code{"BANOVA"}  returned  by  BANOVA.run  function with  a fitted model for an outcome variable regressed on a causal variable, a mediator, and, possibly, moderators and control variables. The outcome variable can follow Normal, T, Poisson, Bernoulli, Binomial, and ordered Multinomial distributions.}

  \item{sol_2}{an  object  of  class  \code{"BANOVA"} returned  by  BANOVA.run  function with  a fitted model for a Normal outcome variable regressed on a causal variable, and, possibly, moderators and control variables.}

  \item{xvar}{a character string which specifies the name of the causal variable used in both models.}
  
  \item{mediator}{a character string which specifies the name of the mediator variable used in the model.}
  
  \item{individual}{logical  indicator  of  whether to output effects for individual units in the analysis (TRUE or FALSE). This analysis requires a multilevel \code{sol_1}.}
  
   \item{return_posterior_samples}{logical indicator of whether posterior samples of mediated effects should be returned (TRUE or FALSE).}
  
  \item{x}{a BANOVA.mediation object}

  \item{\dots}{additional arguments, currently ignored}
  
  \item{multi_samples_beta1_raw_m}{argument for an internal use in the package. Please do not change.}
}

\details{
This function conducts a correlational mediation and moderated mediation analysis (Baron and Kenny 1986; Zao, Lynch and Chen 2010; Zhang, Wedel and Pieters 2008) based on BANOVA models. Based on the samples from posterior distributions, the function calculates the direct effect and indirect effect for which posterior means and 95\% credible intervals are reported. The effect size of the indirect effect is computed as a generalized partial eta-squared. For details about this metric, see the publication of Wedel and Dong (2019).

When the algorithm is calculating the effects of a given causal variable it first identifies all moderators which are interacting with the investigated variable. Based on each interaction, moderated coefficients are computed and summarized in a table. If the causal variable is a part of an interaction term with three or more variables, separate results are computed for each of the moderators and all of their possible combinations. This results in multiple tables with the effects. If a continuous variable is involved in the interaction the effects are evaluated at its mean value, which is zero by default. This is equivalent to omitting the continuous variable from the interaction. 

The function combines the effects of the mediator on the dependent variable with the effect of the causal variable on the mediator in a multiplicative manner to obtain the indirect effect of the treatment. If multiple tables with moderated effects of the mediator or the causal variable on mediator are obtained in the previous steps of the analysis, the indirect effects are computed for each combination of these table. 

For models with a Normal outcome variable, it is possible to interpret the effects as causal by explicitly encoding the causal variable with dummy coding and including an interaction between the causal and mediating variables in the model. For further details, see the publication of MacKinnon et al. (2020).
}
\value{
  \code{BANOVA.mediation} returns an object of class \code{"BANOVA.mediation"}. The returned object is a list containing:
  \item{dir_effects}{tables of the direct effect}
  \item{individual_direct}{a table of the direct effect at the individual level if individual = T and the causal variable is a within-subject variable}
  \item{m1_effects}{tables of the effect of the mediator on the outcome}
  \item{m2_effects}{tables of the effect of the causal variable on the mediator}
  \item{indir_effects}{tables of the indirect effect}
  \item{individual_indirect}{the table of the indirect effect at the individual level if individual = T and the mediator is a within-subject variable}
  \item{effect_size}{a table with the effect size of the mediator}
  \item{xvar}{the name of the causal variable}
  \item{mediator}{the name of the mediating variable}
  \item{individual}{the value of the argument individual (TRUE or FALSE)}

}

\references{ 
Baron, R.M., and Kenny, D.A. (1986) \emph{Moderator Mediator Variables Distinction in Social Psychological Research: Conceptual, Strategic, and Statistical Considerations}, Journal of Personality and Social Psychology, Vol. 51, No. 6, pp. 1173-82.

Zhang, J., Wedel, M. and Pieters, R.G.M.  (2009) \emph{Sales Effects of Attention to Feature Advertisements: A Bayesian Mediation Analysis}, Journal of Marketing Research, Vol.46, No.5, pp. 669-681.

Ying, Y. and MacKinnon, D.P. (2009) \emph{Bayesian Mediation Analysis}, Psychological Methods, Vol. 14, No.4, pp. 301-322.

Zhao, X., John G.L., and Chen, Q. (2010) \emph{Reconsidering Baron and Kenny: Myths and Truths About Mediation Analysis}, Journal of Consumer Research, Vol.37, No.2, pp. 197-206.

Wedel, M., and Dong, C. (2019) \emph{BANOVA: Bayesian Analysis of Variance for Consumer Research}, Journal of Consumer Psychology, Vol. 30, No. 1, pp. 3-23.

MacKinnon, D.P., Valente,  M.J., and Gonzalez, O. (2020) \emph{The correspondence between causal and traditional mediation analysis: the link is the mediator by treatment interaction}, Prevention Science,  Vol. 21, No. 2, pp. 147-157.
}

\examples{
data(condstudy_sub)
\donttest{
# use BANOVA.run based on 'Stan'
model <- BANOVA.model('Normal')
banova_model <- BANOVA.build(model)
res_1 <- BANOVA.run(att~cond+pict, ~type, fit = banova_model, data = condstudy_sub, 
                  id = 'id', iter = 500, thin = 1, chains = 2)
res_2 <- BANOVA.run(pict~cond, ~type, fit = banova_model, data = condstudy_sub, 
                  id = 'id', iter = 500, thin = 1, chains = 2)
# (moderated) mediation
sol <- BANOVA.mediation(res_1, res_2, xvar='cond', mediator='pict')
print(sol)
print(sol$dir_effects)
}
}

