% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BBRecap.R
\name{BBRecap}
\alias{BBRecap}
\title{Bayesian inference for capture-recapture analysis with emphasis on behavioural effect modelling}
\usage{
BBRecap (data,last.column.count=FALSE, neval = 1000, by.incr = 1,
mbc.function = c("standard","markov","counts","integer","counts.integer"),
mod = c("linear.logistic", "M0", "Mb", "Mc", "Mcb", "Mt", "Msubjective.cut",
        "Msubjective"), nsim = 5000, burnin = round(nsim/10),
        nsim.ML = 1000, burnin.ML = round(nsim.ML/10), num.t = 50,
        markov.ord=NULL, prior.N = c("Rissanen","Uniform","one.over.N","one.over.N2"),
        meaningful.mat.subjective = NULL, meaningful.mat.new.value.subjective = NULL,
        z.cut=NULL, output = c("base", "complete", "complete.ML"))
}
\arguments{
\item{data}{can be one of the following:
\enumerate{
  \item an \eqn{M} by \eqn{t} binary matrix/data.frame. In this case the input is interpreted as a matrix whose rows contain individual capture histories for all \eqn{M} observed units
  \item a matrix/data.frame with \eqn{(t+1)} columns. The first \eqn{t} columns contain binary entries corresponding to capture occurrences, while the last column contains non negative integers corresponding to frequencies. This format is allowed only when \code{last.column.count} is set to \code{TRUE}
  \item a \eqn{t}-dimensional array or table representing the counts of the \eqn{2^t} contingency table of binary outcomes
}
\eqn{M} is the number of units captured at least once and \eqn{t} is the number of capture occasions.}

\item{last.column.count}{a logical. In the default case \code{last.column.count=FALSE} each row of the input argument \code{data} represents the complete capture history for each observed unit. When \code{last.column.count} is set to \code{TRUE} in each row the first \eqn{t} entries represent one of the observed complete capture histories and the last entry in the last column is the number of observed units with that capture history}

\item{neval}{a positive integer. \code{neval} is the number of values of the population size \eqn{N} where the posterior is evaluated starting from \eqn{M}. The default value is \code{neval}=1000.}

\item{by.incr}{a positive integer. \code{nsim} is the number of iterations for the Metropolis-within-Gibbs algorithm which allows the approximation of the posterior. It is considered only if \code{mod} is \code{"linear.logistic"} or \code{"Msubjective"}. In the other cases closed form evaluation of the posterior is available up to a proportionality constant. The default value is \code{nsim=10000.}}

\item{mbc.function}{a character string with possible entries (see Alunni Fegatelli (2013) for further details)
\enumerate{
  \item \code{"standard"} meaningful behavioural covariate in [0,1] obtained through the normalized binary representation of integers relying upon partial capture history
  \item \code{"markov"} slight modification of \code{"standard"} providing consistency with arbitrary Markov order models when used in conjunction with the options \code{"Msubjective"} and \code{z.cut}.
  \item \code{"counts"} covariate in [0,1] obtained by normalizing the integer corresponding to the sum of binary entries i.e. the number of previous captures
  \item \code{"integer"} un-normalized integer corresponding to the binary entries of the partial capture history
  \item \code{"counts.integer"} un-normalized covariate obtained as the sum of binary entries i.e. the number of previous captures
  }}

\item{mod}{a character. \code{mod} represents the behavioural model considered for the analysis. \code{mod="linear.logistic"} is the model proposed in Alunni Fegatelli (2013) based on the meaningful behavioural covariate. \code{mod="M0"} is the most basic model where no effect is considered and all capture probabilities are the same. \code{mod="Mb"} is the classical behavioural model where the capture probability varies only once when first capture occurs. Hence it represents an \emph{enduring} effect to capture. \code{mod="Mc"} is the \emph{ephemeral} behavioural Markovian model originally introduced in Yang and Chao (2005) and subsequently extended in Farcomeni (2011) and reviewed in Alunni Fegatelli and Tardella (2012) where capture probability depends only on the capture status (captured or uncaptured) in the previous \code{k=markov.ord} occasions. \code{mod="Mcb"} is an extension of Yang and Chao's model (2005); it considers both \emph{ephemeral} and \emph{enduring} effect to capture. \code{mod="Mt"} is the standard temporal effect with no behavioural effect. \code{mod="Msubjective.cut"} is an alternative behavioural model obtained through a specific cut on the meaningful behavioural covariate interpreted as memory effect. \code{mod="Msubjective"} is a customizable (subjective) behavioural model within the linear logistic model framework requiring the specification of the two additional arguments: the first one is \code{meaningful.mat.subjective} and contains an \eqn{M} by \eqn{t} matrix of ad-hoc meaningful covariates depending on previous capture history; the second one is \code{meaningful.mat.new.value.subjective} and contains a vector of length \eqn{t} corresponding to meaningful covariates for a generic uncaptured unit. The default value for \code{mod} is \code{"linear.logistic".}}

\item{nsim}{a positive integer. \code{nsim} is the number of iterations for the Metropolis-within-Gibbs algorithm which allows the approximation of the posterior. It is considered only if \code{mod} is \code{"linear.logistic"} or \code{"Msubjective"}. In the other cases closed form evaluation of the posterior is available up to a proportionality constant. The default value is \code{nsim=10000.}}

\item{burnin}{a positive integer. \code{burnin} is the initial number of MCMC samples discarded. It is considered only if \code{mod} is \code{"linear.logistic"}  or \code{"Msubjective"}. The default value for \code{burnin} is \code{round(nsim/10).}}

\item{nsim.ML}{a positive integer. Whenever MCMC is needed \code{nsim.ML} is the number of iterations used in the marginal likelihood estimation procedure via power posterior method of Friel and Pettit (2008)}

\item{burnin.ML}{a positive integer. Whenever MCMC is needed \code{burnin.ML} is the initial number of samples discarded for marginal likelihood estimation via power-posterior approach. The default value is \code{burnin.ML} is \code{round(nsim/10)}.}

\item{num.t}{a positive integer. Whenever MCMC is needed \code{num.t} is the number of powers used in the power posterior approximation method for the marginal likelihood evaluation.}

\item{markov.ord}{a positive integer. \code{markov.ord} is the order of Markovian model \eqn{M_c} or \eqn{M_{cb}}. It is considered only if \code{mod="Mc"} or \code{mod="Mcb"}.}

\item{prior.N}{a character. \code{prior.N} is the label for the prior distribution for \eqn{N}. When \code{prior.N} is set to \code{"Rissanen"} (default) the Rissanen prior is used as a prior on \eqn{N}. This distribution was first proposed in Rissanen 1983 as a universal prior on integers. \code{prior.N="Uniform"} stands for a prior on \eqn{N} proportional to a constant value. \code{prior.N="one.over.N"} stands for a prior on \eqn{N} proportional to \eqn{1/N}. \code{prior.N="one.over.N2"} stands for a prior on \eqn{N} proportional to \eqn{1/N^2}.}

\item{meaningful.mat.subjective}{\code{M x t} matrix containing  numerical covariates to be used for a customized logistic model approach}

\item{meaningful.mat.new.value.subjective}{\code{1 x t} numerical vector corresponding to auxiliary covariate to be considered for unobserved unit}

\item{z.cut}{numeric vector. \code{z.cut} is a vector containing the cut point for the memory effect covariate. It is considered only if \code{mod="Msubjective.cut"}}

\item{output}{a character. \code{output} selects the kind of output from a very basic summary info on the posterior output (point and interval estimates for the unknown \eqn{N}) to more complete details including MCMC simulations for all parameters in the model when appropriate.}
}
\value{
\item{Model}{model considered}
  \item{Prior}{prior distribution for \eqn{N}}
  \item{N.hat.mean}{posterior mean for \eqn{N}}
  \item{N.hat.median}{posterior median for \eqn{N}}
  \item{N.hat.mode}{posterior mode for \eqn{N}}
  \item{N.hat.RMSE}{minimizer of a specific loss function connected with the Relative Mean Square Error}
  \item{HPD.N}{\eqn{95 \%} highest posterior density interval estimate for \eqn{N}}
  \item{log.marginal.likelihood}{log marginal likelihood}
  \item{N.range}{values of N considered}
  \item{posterior.N}{values of the posterior distribution for each N considered}
  \item{z.matrix}{meaningful behavioural covariate matrix for the observed data}
  \item{vec.cut}{cut point used to set up meaningful partitions the set of the partial capture histories according to the value of the value of the meaningful behavioural covariate}
  \item{N.vec}{simulated values from the posterior marginal distribution of N}
  \item{mean.a0}{posterior mean of the parameter a0}
  \item{hpd.a0}{highest posterior density interval estimate of the parameter a0}
  \item{a0.vec}{simulated values from the posterior marginal distribution of a0}
  \item{mean.a1}{posterior mean of the parameter a1}
  \item{hpd.a1}{highest posterior density interval estimate of the parameter a1}
  \item{a1.vec}{simulated values from the posterior marginal distribution of a1}
}
\description{
Bayesian inference for a large class of discrete-time capture-recapture models under closed population with special emphasis on behavioural effect modelling including also the \emph{meaningful behavioral covariate} approach proposed in Alunni Fegatelli (2013) [PhD thesis]. Many of the standard classical models such as \eqn{M_0}, \eqn{M_b}, \eqn{M_{c_1}}, \eqn{M_t} or \eqn{M_{bt}} can be regarded as particular instances of the aforementioned approach. Other flexible alternatives can be fitted through a careful choice of a meaningful behavioural covariate and a possible partition of its admissible range.
}
\details{
Independent uniform distributions are considered as default prior for the nuisance parameters. If \code{model="linear.logistic"} or \code{model="Msubjective"} and \code{output="complete.ML"}  the marginal likelihood estimation is performed through the \emph{power posteriors method} suggested in Friel and Pettit (2008). In that case the \code{BBRecap} procedure is computing intensive for high values of \code{neval} and \code{nsim}.
}
\examples{

\dontrun{
data(greatcopper)

mod.Mb=BBRecap(greatcopper,mod="Mb")
str(mod.Mb)
}

}
\references{
Otis D. L., Burnham K. P., White G. C, Anderson D. R. (1978) Statistical Inference From Capture Data on Closed Animal Populations, Wildlife Monographs.

Yang H.C., Chao A. (2005) Modeling animals behavioral response by Markov chain models for capture-recapture experiments, Biometrics 61(4), 1010-1017

N. Friel and A. N. Pettitt. Marginal likelihood estimation via power posteriors. Journal of the Royal Statistical Society: Series B (Statistical Methodology), 70(3):589, 607--2008

Farcomeni A. (2011) Recapture models under equality constraints for the conditional capture probabilities. Biometrika 98(1):237--242

Alunni Fegatelli, D. and Tardella, L. (2012) Improved inference on capture recapture models with behavioural effects. Statistical Methods & Applications Applications Volume 22, Issue 1, pp 45-66 10.1007/s10260-012-0221-4

Alunni Fegatelli D. (2013) New methods for capture-recapture modelling with behavioural response and individual heterogeneity. http://hdl.handle.net/11573/918752
}
\seealso{
\code{\link{BBRecap.custom.part}}, \code{\link{LBRecap}}
}
\author{
Danilo Alunni Fegatelli and Luca Tardella
}
\keyword{Bayesian_inference}
\keyword{Behavioural_models}
