% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hd.R
\name{hd}
\alias{hd}
\alias{hd.bgvar.irf}
\title{Historical Decomposition}
\usage{
hd(x, R=NULL, verbose=TRUE)
}
\arguments{
\item{x}{an item fitted by \code{IRF}.}

\item{R}{If \code{NULL} and the \code{irf.bgvar} object has been fitted via sign restrictions, the rotation matrix is used that minimizes the distance to the median impulse responses at the posterior median.}

\item{verbose}{If set to \code{FALSE} it suppresses printing messages to the console.}
}
\value{
Returns a list with the following objects \itemize{
\item{\code{hd_array}}{ is a three-dimensional array with the first dimension referring to the K time series, the second to the T observations and the third dimensions containing the contribution of the shocks in explaining historically deviations in the time series from their trend. The third dimension is K+3, since the last three entries contain the contributions of the constant, the initial condition and a residual component that the contributions sum up to the original time series. If a trend i specified in the model the third dimension is K+3 with trend ordered after the constant.}
\item{\code{struc.shcok}}{ contains the structural shock.}
\item{\code{x}}{ is a matrix object that contains the original time series, which is of dimension K times (T-plag).}
}
}
\description{
A function that calculates historical decomposition (HD) of the time series and the structural error.
}
\details{
To save computational time as well as due to storage limits, both functions are based on the posterior median (as opposed to calculating HDs and the structural error for each draw of the MCMC chain). In case the shock has been identified via sign restrictions, a rotation matrix has to be selected to calculate both statistics. If not specified otherwise (via \code{R}), the algorithm searches for 50 rotation matrices that fulfill the sign restrictions at the \emph{posterior median} of the coefficients and then singles out the rotation matrix that minimizes the distance to the median of the impulse responses as suggested in Fry and Pagan (2011).
}
\examples{
\dontshow{
library(BGVAR)
data(eerData)
cN<-c("EA","US","UK")
eerData<-eerData[cN]
W.trade0012<-apply(W.trade0012[cN,cN],2,function(x)x/rowSums(W.trade0012[cN,cN]))
model.ssvs.eer<-bgvar(Data=eerData,W=W.trade0012,draws=100,burnin=100,plag=1,
                      prior="SSVS",thin=1,eigen=TRUE)
shocks<-list();shocks$var="stir";shocks$cN<-"US";shocks$ident="chol";shocks$scal=-100
irf.chol.us.mp <- irf(model.ssvs.eer,shock=shocks,n.ahead=48)
HD <- hd(irf.chol.us.mp)
}
\donttest{
set.seed(571)
library(BGVAR)
data(eerData)
model.ssvs.eer<-bgvar(Data=eerData,W=W.trade0012,draws=100,burnin=100,plag=1,
                      prior="SSVS",thin=1,eigen=TRUE)
# US monetary policy shock
shocks<-list();shocks$var="stir";shocks$cN<-"US";shocks$ident="chol";shocks$scal=-100
irf.chol.us.mp <- irf(model.ssvs.eer,shock=shocks,n.ahead=48)

HD <- hd(irf.chol.us.mp)
# summing them up should get you back the original time series
org.ts<-apply(HD$hd_array,c(1,2),sum)
matplot(cbind(HD$xglobal[,1],org.ts[,1]),type="l",ylab="")
legend("bottomright",c("hd series","original"),col=c("black","red"),lty=c(1,2),bty="n")
}
}
\references{
Fry, R. and A. Pagan (2011) \emph{Sign restrictions in Structural Vector Autoregressions: A Critical Review}. Journal of Economic Literature, Vol. 49(4), pp. 938-960.
}
\seealso{
\code{\link{bgvar}} and \code{\link{irf}}.
}
\author{
Maximilian Boeck, Martin Feldkircher, Florian Huber
}
