% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/10_bvar.R
\name{bvar}
\alias{bvar}
\title{Hierarchical Bayesian Vector Autoregression}
\usage{
bvar(data, lags, n_draw = 10000, n_burn = 5000, n_thin = 1,
  priors = bv_priors(), mh = bv_mh(), fcast = bv_fcast(),
  irf = bv_irf(), verbose = TRUE, ...)
}
\arguments{
\item{data}{Numeric matrix or dataframe. Observations need to be ordered
from earliest to latest one.}

\item{lags}{Integer scalar. Number of lags to apply to the data.}

\item{n_draw}{Integer scalar. Number of total iterations for the model
to cycle through.}

\item{n_burn}{Integer scalar. Number of iterations to discard.}

\item{n_thin}{Integer scalar. Option to reduce the number of stored
iterations to every \emph{n_thin}th one. The number of saved iterations will
thus be \code{(n_draw - n_burn) / n_thin}.}

\item{priors}{\code{bv_priors} object containing priors and their settings.
See \code{\link{bv_priors}}.}

\item{mh}{\code{bv_metropolis} object with settings regarding the acceptance
rate of the Metropolis-Hastings step. See \code{\link{bv_mh}}.}

\item{fcast}{\code{bv_fcast} object. Forecast options set with
\code{\link{bv_fcast}}. May be set to \code{NULL} to skip forecasting.}

\item{irf}{\code{bv_irf} object. Options regarding impulse responses, set via
\code{\link{bv_irf}}. May be set to \code{NULL} to skip the calculation of
impulse responses.}

\item{verbose}{Logical scalar. Whether to print intermediate results and
progress.}

\item{...}{Not used.}
}
\value{
Returns a \code{bvar} object with the following elements:
\itemize{
  \item \code{beta} - Numeric array with saved draws from the posterior
  distribution of the VAR coefficients.
  \item \code{sigma} - Numeric array with saved draws from the posterior
  distribution of the model's vcov-matrix.
  \item \code{hyper} - Numeric matrix with saved draws from the posterior
  distributions of hyperparameters of hierarchical priors.
  \item \code{ml} - Numeric vector with the value of the posterior marginal
  likelihood corresponding to each draw of hyperparameters and associated
  VAR coefficients.
  \item \code{optim} - List with outputs from \code{\link[stats]{optim}}.
  \item \code{prior} - \code{bv_priors} object. See \code{\link{bv_priors}}.
  \item \code{call} - Call to the function. See \code{\link{match.call}}.
  \item \code{meta} - List with meta information such as number of variables,
  accepted draws, number of iterations et cetera.
  \item \code{fcast} - \code{bvar_fcast} object with posterior forecast
  draws as well as the forecast's setup from \emph{fcast}.
  \item \code{irf} - \code{bvar_irf} object with posterior impulse response
  draws, as well as the impulse response setup from \emph{irf}.
}
}
\description{
Hierarchical Bayesian estimation of VAR models in the fashion of Giannone et
al. (2015).
}
\examples{
# Access the fred_qd dataset and transform it
data("fred_qd")
data <- fred_qd[, c("CPIAUCSL", "UNRATE", "FEDFUNDS")]
data[5:nrow(data), 1] <- diff(log(data[, 1]), lag = 4) * 100
data <- data[5:nrow(data), ]

# Compute VAR using 2 lags and a ridiculously low number of draws
x <- bvar(
  data = data, lags = 1,
  n_draw = 500, n_burn = 400, n_thin = 2, verbose = FALSE
)

# Plot various outputs generated
\donttest{
plot(x)
bv_plot_fcast(x)
bv_plot_irf(x)
}
}
\references{
Giannone, D., Lenza, M., & Primiceri, G. E. (2015). Prior Selection for Vector Autoregressions. Review of Economics and Statistics, 97, 436-451. \url{https://doi.org/10.1162/REST_a_00483}
}
