% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/10_bvar.R
\name{bvar}
\alias{bvar}
\title{Hierarchical Bayesian vector autoregression}
\usage{
bvar(
  data,
  lags,
  n_draw = 10000L,
  n_burn = 5000L,
  n_thin = 1L,
  priors = bv_priors(),
  mh = bv_mh(),
  fcast = NULL,
  irf = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Numeric matrix or dataframe. Note that observations are expected
to be ordered from earliest to latest, and variables in the columns.}

\item{lags}{Integer scalar. Lag order of the model.}

\item{n_draw, n_burn}{Integer scalar. The number of iterations to (a) cycle
through and (b) burn at the start.}

\item{n_thin}{Integer scalar. Every \emph{n_thin}'th iteration is stored.
For a given memory requirement thinning reduces autocorrelation, while
increasing effective sample size.}

\item{priors}{Object from \code{\link{bv_priors}} with prior settings.
Used to adjust the Minnesota prior, add custom dummy priors, and choose
hyperparameters for hierarchical estimation.}

\item{mh}{Object from \code{\link{bv_mh}} with settings for the
Metropolis-Hastings step. Used to tune automatic adjustment of the
acceptance rate within the burn-in period, or manually adjust the proposal
variance.}

\item{fcast}{Object from \code{\link{bv_fcast}} with forecast settings.
Options include the horizon and settings for conditional forecasts i.e.
scenario analysis.
May also be calculated ex-post using \code{\link{predict.bvar}}.}

\item{irf}{Object from \code{\link{bv_irf}} with settings for the calculation
of impulse responses and forecast error variance decompositions. Options
include the horizon and different identification schemes.
May also be calculated ex-post using \code{\link{irf.bvar}}.}

\item{verbose}{Logical scalar. Whether to print intermediate results and
progress.}

\item{...}{Not used.}
}
\value{
Returns a list of class \code{bvar} with the following elements:
\itemize{
  \item \code{beta} - Numeric array with draws from the posterior of the VAR
    coefficients. Also see \code{\link{coef.bvar}}.
  \item \code{sigma} - Numeric array with draws from the posterior of the
    variance-covariance matrix. Also see \code{\link{vcov.bvar}}.
  \item \code{hyper} - Numeric matrix with draws from the posterior of the
    hierarchically treated hyperparameters.
  \item \code{ml} - Numeric vector with the marginal likelihood (with respect
  to the hyperparameters), that determines acceptance probability.
  \item \code{optim} - List with outputs of \code{\link[stats]{optim}},
    which is used to find starting values for the hyperparameters.
  \item \code{prior} - Prior settings from \code{\link{bv_priors}}.
  \item \code{call} - Call to the function. See \code{\link{match.call}}.
  \item \code{meta} - List with meta information. Includes the number of
    variables, accepted draws, number of iterations, and data.
  \item \code{variables} - Character vector with the column names of
    \emph{data}. If missing, variables are named iteratively.
  \item \code{explanatories} - Character vector with names of explanatory
    variables. Formatting is akin to: \code{"FEDFUNDS-lag1"}.
  \item \code{fcast} - Forecasts from \code{\link{predict.bvar}}.
  \item \code{irf} - Impulse responses from \code{\link{irf.bvar}}.
}
}
\description{
Used to estimate hierarchical Bayesian Vector Autoregression (VAR) models in
the fashion of Giannone, Lenza and Primiceri (2015).
Priors are adjusted and added via \code{\link{bv_priors}}.
The Metropolis-Hastings step can be modified with \code{\link{bv_mh}}.
}
\details{
The model can be expressed as:
\deqn{y_t = a_0 + A_1 y_{t-1} + ... + A_p y_{t-p} + \epsilon_t}{y_t = a_0 +
A_1 y_{t-1} + ... + A_p y_{t-p} + e_t}
See Kuschnig and Vashold (2021) and Giannone, Lenza and Primiceri (2015)
for further information.
Methods for a \code{bvar} object and its derivatives can be used to:
\itemize{
  \item predict and analyse scenarios;
  \item evaluate shocks and the variance of forecast errors;
  \item visualise forecasts and impulse responses, parameters and residuals;
  \item retrieve coefficents and the variance-covariance matrix;
  \item calculate fitted and residual values;
}
Note that these methods generally work by calculating quantiles from the
posterior draws. The full posterior may be retrieved directly from the
objects. The function \code{\link[utils]{str}} can be very helpful for this.
}
\examples{
# Access a subset of the fred_qd dataset
data <- fred_qd[, c("CPIAUCSL", "UNRATE", "FEDFUNDS")]
# Transform it to be stationary
data <- fred_transform(data, codes = c(5, 5, 1), lag = 4)

# Estimate a BVAR using one lag, default settings and very few draws
x <- bvar(data, lags = 1, n_draw = 1000L, n_burn = 200L, verbose = FALSE)

# Calculate and store forecasts and impulse responses
predict(x) <- predict(x, horizon = 8)
irf(x) <- irf(x, horizon = 8, fevd = FALSE)

\dontrun{
# Check convergence of the hyperparameters with a trace and density plot
plot(x)
# Plot forecasts and impulse responses
plot(predict(x))
plot(irf(x))
# Check coefficient values and variance-covariance matrix
summary(x)
}
}
\references{
Giannone, D. and Lenza, M. and Primiceri, G. E. (2015) Prior Selection for
  Vector Autoregressions. \emph{The Review of Economics and Statistics},
  \bold{97:2}, 436-451, \doi{10.1162/REST_a_00483}.

  Kuschnig, N. and Vashold, L. (2021) BVAR: Bayesian Vector Autoregressions
  with Hierarchical Prior Selection in R.
  \emph{Journal of Statistical Software}, \bold{14}, 1-27,
  \doi{10.18637/jss.v100.i14}.
}
\seealso{
\code{\link{bv_priors}}; \code{\link{bv_mh}};
\code{\link{bv_fcast}}; \code{\link{bv_irf}};
\code{\link{predict.bvar}}; \code{\link{irf.bvar}}; \code{\link{plot.bvar}};
}
\author{
Nikolas Kuschnig, Lukas Vashold
}
\keyword{BVAR}
\keyword{MCMC}
\keyword{Metropolis-Hastings}
\keyword{hierarchical}
\keyword{priors}
