% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyzeTS.R
\name{analyzeTS}
\alias{analyzeTS}
\alias{reportTS}
\alias{simulateTS}
\title{The Functions analyzeTS, reportTS, and simulateTS}
\usage{
analyzeTS(
  TS,
  season = "month",
  dist = "ggamma",
  acsID = "weibull",
  norm = "N1",
  n.points = 30,
  lag.max = 30,
  constrain = FALSE,
  opts = NULL
)

reportTS(aTS, method = "dist")

simulateTS(aTS, from = NULL, to = NULL)
}
\arguments{
\item{TS}{time series in format - date, value}

\item{season}{name of the season (e.g. month, week)}

\item{dist}{name of the distribution to be fitted}

\item{acsID}{ID of the autocorrelation structure to be fitted}

\item{norm}{norm used for distribution fitting - id ('N1', 'N2', 'N3', 'N4')}

\item{n.points}{number of points to be subsetted from ecdf}

\item{lag.max}{max lag for the empirical autocorrelation structure}

\item{constrain}{logical - constrain shape2 parametes for finite tails}

\item{opts}{minimization options}

\item{aTS}{analyzed timeseries}

\item{method}{report method - 'dist' for distribution fits, 'acs' for ACS fits and 'stat' for basic statistical report}

\item{from}{starting date/time of the simulation}

\item{to}{end date/time of the simulation}
}
\description{
Provide a complete set of tools to make time series analysis a piece of cake - analyzeTS() automatically performs seasonal analysis, fits distributions and correlation structures, reportTS provides visualizations of the fitted distributions and correlation structures, and a table with the values of the fitted parameters and basic descriptive statistics, simulateTS automatically takes the results of the analyseTS and generates syntetic ones.
}
\details{
In practice, we usually want to simulate a natural process using some sampled time series. To generate a synthetic time series with similar characteristics to the observed values, we have to determine marginal distribution, autocorrelation structure and probability zero for each individual month. This can is done by fitting distributions and autocorrelation structures with analyzeTS(). Result can be checked with reportTS(). Syynthetic time series with the same statistical properties can be produced with simulateTS().

Recomended distributions for variables:
\itemize{
\item \emph{precipitation}: ggamma (Generalized Gamma), burr### (Burr type)
\item \emph{streamflow}: ggamma (Generalized Gamma), burr### (Burr type)
\item \emph{relative humidity}: beta
\item \emph{temperature}: norm (Normal distribution)
}
}
\examples{

library(CoSMoS)

## Load data included in the package
## (to find out more about the data use ?precip)
data('precip')
\donttest{
## Fit seasonal ACSs and distributions to the data
a <- analyzeTS(precip)

reportTS(a, 'dist') ## show seasonal distribution fit
reportTS(a, 'acs') ## show seasonal ACS fit
reportTS(a, 'stat') ## display basic descriptive statisctics

######################################
## 'duplicate' analyzed time series ##
sim <- simulateTS(a)

## plot the result
precip[, id := 'observed']
sim[, id := 'simulated']

dta <- rbind(precip, sim)

ggplot(dta) +
  geom_line(aes(x = date, y = value)) +
  facet_wrap(~id, ncol = 1) +
  theme_classic()

################################################
## or simulate timeseries of different length ##
sim <- simulateTS(a,
                  from = as.POSIXct('1978-12-01 00:00:00'),
                  to = as.POSIXct('2008-12-01 00:00:00'))

## and plot the result
precip[, id := 'observed']
sim[, id := 'simulated']

dta <- rbind(precip, sim)

ggplot(dta) +
  geom_line(aes(x = date, y = value)) +
  facet_wrap(~id, ncol = 1) +
  theme_classic()
}
\dontshow{
## test for one month to make it fast
precip <- precip[between(date, as.POSIXct('1990-1-01', format('\%Y-\%m-\%d'), tz = 'America/Regina'), as.POSIXct('1990-1-5', format('\%Y-\%m-\%d'), tz = 'America/Regina'))]
a <- analyzeTS(precip, opts = list('algorithm' = 'NLOPT_LN_NELDERMEAD',
                                   'maxeval' = 10))
}

}
