% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crit_AL.R
\name{crit_AL}
\alias{crit_AL}
\title{Expected Augmented Lagrangian Improvement}
\usage{
crit_AL(
  x,
  model.fun,
  model.constraint,
  equality = FALSE,
  critcontrol = NULL,
  type = "UK"
)
}
\arguments{
\item{x}{either a vector representing the design or the design AND slack variables (see details)}

\item{model.fun}{object of class \code{\link[DiceKriging]{km}} correspostnding to the objective function,
or, if the objective function is fast-to-evaluate, a \code{\link[DiceOptim]{fastfun}} object,}

\item{model.constraint}{either one or a list of objects of class \code{\link[DiceKriging]{km}}, one for each constraint function,}

\item{equality}{either \code{FALSE} if all constraints are for inequalities, or a vector of Booleans indicating which are equalities}

\item{critcontrol}{optional list with the following arguments:
\itemize{
 \item{\code{slack}}{: logical. If TRUE, slack variables are used for inequality constraints (see Details)}
 \item{\code{rho}}{: penalty term (scalar),}
 \item{\code{lambda}}{: Lagrange multipliers (vector of size the number of constraints),}
 \item{\code{elit}}{: logical. If TRUE, sets the criterion to zero for all x's not improving the objective function}
 \item{\code{n.mc}}{: number of Monte-Carlo drawings used to evaluate the criterion (see Details)}
 \item{\code{nt}}{: number of discretization points for the WNCS distribution (see Details)}
 \item \code{tolConstraints}, an optional vector giving a tolerance (> 0) for each of the constraints (equality or inequality).
It is highly recommended to use it when there are equality constraints since the default tolerance of 0.05 in such case might not be suited.
 }
 
Options for the \code{\link[DiceOptim]{checkPredict}} function: \code{threshold} (\code{1e-4}) and \code{distance} (\code{covdist}) 
are used to avoid numerical issues occuring when adding points too close to the existing ones.}

\item{type}{"\code{SK}" or "\code{UK}" (by default), depending whether uncertainty related to trend estimation 
has to be taken into account.}
}
\value{
The Expected Augmented Lagrangian Improvement at \code{x}.
}
\description{
Computes the Expected Augmented Lagrangian Improvement at current location, with our without slack variables. Depending on the cases,
the computation is either analytical (very fast), based on MC integration (slow) or on the CDF of a weighted sum of non-central
chi-square (WNCS) variates (intermediate)
}
\details{
The AL can be used with or without the help of slack variables for the inequality constraints. 
If \code{critcontrol$slack=FALSE}:
With a single constraint (inequality or equality) and a fast objective, a very fast formula is
used to compute the criterion (recommended setting). 
Otherwise, an MC estimator of the criterion is used, which is much more costly. The argument 
\code{critcontrol$n.mc} tunes the precision of the estimator.
On both cases \code{x} must be of size \code{d}.

If \code{critcontrol$slack=TRUE}:
Slack variables are used to handle the inequality constraints. 
They can be provided directly through \code{x}, which should be of size \code{d+} the number of inequality constraints.
The last values of \code{x} are slack variables scaled to [0,1].

If \code{x} is of size \code{d}, estimates of optimal slack variable are used.\cr
}
\examples{
#---------------------------------------------------------------------------
# Expected Augmented Lagrangian Improvement surface with one inequality constraint,
# fast objective
#---------------------------------------------------------------------------
\donttest{
set.seed(25468)
library(DiceDesign)

n_var <- 2 
fun.obj <- goldsteinprice
fun.cst <- function(x){return(-branin(x) + 25)}
n.grid <- 31
test.grid <- expand.grid(X1 = seq(0, 1, length.out = n.grid), X2 = seq(0, 1, length.out = n.grid))
obj.grid <- apply(test.grid, 1, fun.obj)
cst.grid <- apply(test.grid, 1, fun.cst)
n.init <- 15 
design.grid <- round(maximinESE_LHS(lhsDesign(n.init, n_var, seed = 42)$design)$design, 1)
obj.init <- apply(design.grid, 1, fun.obj)
cst.init <- apply(design.grid, 1, fun.cst)
model.constraint <- km(~., design = design.grid, response = cst.init)
model.fun <- fastfun(fun.obj, design.grid)
AL_grid <- apply(test.grid, 1, crit_AL, model.fun = model.fun,
                  model.constraint = model.constraint)

filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), nlevels = 50,
               matrix(AL_grid, n.grid), main = "Expected AL Improvement",
               xlab = expression(x[1]), ylab = expression(x[2]), color = terrain.colors, 
               plot.axes = {axis(1); axis(2);
                            points(design.grid[,1], design.grid[,2], pch = 21, bg = "white")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                            matrix(obj.grid, n.grid), nlevels = 10,
                                   add=TRUE,drawlabels=TRUE, col = "black")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                            matrix(cst.grid, n.grid), level = 0, add=TRUE,
                                   drawlabels=FALSE,lwd=1.5, col = "red")
                            }
              )
}
#---------------------------------------------------------------------------
# Expected AL Improvement surface with one inequality and one equality constraint,
# using slack variables
#---------------------------------------------------------------------------
\donttest{
set.seed(25468)
library(DiceDesign)

n_var <- 2 
fun.obj <- goldsteinprice
fun.cstineq <- function(x){return(3/2 - x[1] - 2*x[2] - .5*sin(2*pi*(x[1]^2 - 2*x[2])))}
fun.csteq <- function(x){return(branin(x) - 25)}
n.grid <- 51
test.grid <- expand.grid(X1 = seq(0, 1, length.out = n.grid), X2 = seq(0, 1, length.out = n.grid))
obj.grid <- apply(test.grid, 1, fun.obj)
cstineq.grid <- apply(test.grid, 1, fun.cstineq)
csteq.grid <- apply(test.grid, 1, fun.csteq)
n.init <- 25 
design.grid <- round(maximinESE_LHS(lhsDesign(n.init, n_var, seed = 42)$design)$design, 1)
obj.init <- apply(design.grid, 1, fun.obj)
cstineq.init <- apply(design.grid, 1, fun.cstineq)
csteq.init <- apply(design.grid, 1, fun.csteq)
model.fun <- km(~., design = design.grid, response = obj.init)
model.constraintineq <- km(~., design = design.grid, response = cstineq.init)
model.constrainteq <- km(~., design = design.grid, response = csteq.init)

models.cst <- list(model.constraintineq, model.constrainteq)
 
AL_grid <- apply(test.grid, 1, crit_AL, model.fun = model.fun, model.constraint = models.cst,
                  equality = c(FALSE, TRUE), critcontrol = list(tolConstraints = c(0.05, 3),
                  slack=TRUE))

filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), nlevels = 50,
               matrix(AL_grid, n.grid), main = "Expected AL Improvement",
               xlab = expression(x[1]), ylab = expression(x[2]), color = terrain.colors, 
               plot.axes = {axis(1); axis(2);
                            points(design.grid[,1], design.grid[,2], pch = 21, bg = "white")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                            matrix(obj.grid, n.grid), nlevels = 10,
                                   add=TRUE,drawlabels=TRUE, col = "black")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                            matrix(cstineq.grid, n.grid), level = 0, add=TRUE,
                                   drawlabels=FALSE,lwd=1.5, col = "red")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid),
                            matrix(csteq.grid, n.grid), level = 0, add=TRUE,
                                   drawlabels=FALSE,lwd=1.5, col = "orange")
                            }
              )
}
}
\references{
R.B. Gramacy, G.A. Gray, S. Le Digabel, H.K.H Lee, P. Ranjan, G. Wells, Garth, and S.M. Wild (2014+),
Modeling an augmented Lagrangian for improved blackbox constrained optimization,
\emph{arXiv preprint arXiv:1403.4890}.
}
\seealso{
\code{\link[DiceOptim]{EI}} from package DiceOptim, \code{\link[DiceOptim]{crit_EFI}}, \code{\link[DiceOptim]{crit_SUR_cst}}.
}
\author{
Victor Picheny 

Mickael Binois
}
