% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Disequilibrium.R
\name{DE}
\alias{DE}
\title{Market in Disequilibrium Model}
\usage{
DE(formula, data, subset = NULL, par = NULL, control = list())
}
\arguments{
\item{formula}{An object of class \link[Formula]{Formula}: a symbolic
description of the model to be fitted. The details of model specification
are given under 'Details'.}

\item{data}{A required data frame containing the variables provided in \code{formula}.}

\item{subset}{An optional numeric vector specifying a subset of observations to be used in the fitting process.}

\item{par}{A vector of initial values for the parameters for which optimal values are to be found.
The order of the parameters is the coefficients of equation 1, the coefficients of equation 2, the variance for equation 1,
the covariance of equations 1 and 2, and the variance of equation 2.  The default is 0 for all parameters, except the variance
parameters, which are set to 1.}

\item{control}{A list of control parameters. See 'Details'.}
}
\value{
An object of class 'DE' is returned as a list with the following components:

\describe{

\item{par}{The set of parameter maximum likelihood estimates.}

\item{value}{The negative of the log likelihood evaluated at \code{par}.}

\item{counts}{A two-element integer vector giving the number of calls to \code{fn} and \code{gr} respectively.
 This excludes those calls needed to compute the Hessian, if requested, and any calls to \code{fn} to compute a finite-difference
 approximation to the gradient.}

\item{convergence}{An integer code. '0' indicates successful completion.}

\item{message}{A character string giving any additional information returned by the optimizer, or \code{NULL}.}

\item{Sigma}{The estimated covariance matrix.  This is a subset of \code{par}.}

\item{Xbeta1}{The predicted outcome for each observation in equation 1 using the parameters estimated in \code{par}.}

\item{Xbeta2}{The predicted outcome for each observation in equation 2 using the parameters estimated in \code{par}.}

\item{Y}{A vector of the response (quantity) values.}

\item{X}{A list of the design matricies for the two equations.}

\item{hessian}{A numerical approximation to the Hessian matrix of the likelihood function at the estimated parameter values.
 The Hessian is returned even if it is not requested.}

\item{vcov}{The variance covariance matrix of the estimates in \code{par}.}

\item{MaskRho}{The value of \code{MaskRho} used.}

}

The object of class 'DE' has the following attributes:

\describe{

\item{originalNamesX1}{The original names of the covariates for equation 1 specified in \code{formula}.}

\item{originalNamesX2}{The original names of the covariates for equation 2 specified in \code{formula}.}

\item{namesX1}{The names of the covariates for equation 1 to be used in the \code{summary.DE} function.
 This is equivalent to \code{paste0(originalNamesX1, Equation1Name)}.}

\item{namesX2}{The names of the covariates for equation 2 to be used in the \code{summary.DE} function.
 This is equivalent to \code{paste0(originalNamesX2, Equation2Name)}.}

\item{namesSigma}{The names of the variance-covariance matrix parameters to be used in the \code{summary.DE} function.}

\item{Equation1Name}{The user-specified name of equation 1.}

\item{Equation2Name}{The user-specified name of equation 2.}

\item{betaN1}{The number of coefficient and slope parameters to be estimated in equation 1.}

\item{betaN2}{The number of coefficient and slope parameters to be estimated in equation 2.}

}
}
\description{
\code{DE} estimates a market in disequilibrium model.

The market in disequilibrium model is defined as follows.
Let \eqn{i} denote the \eqn{i}th observation which takes values from \eqn{1}
to \eqn{N}, \eqn{X_1}{X[1]} be a covariate matrix of dimension
\eqn{N \times k_1}{N x k[1]}, \eqn{X_2}{X[2]} be a covariate matrix of
dimension \eqn{N \times k_2}{N x k[2]}, \eqn{X_{1i}}{X[1i]} be the \eqn{i}th
row of \eqn{X_1}{X[1]}, \eqn{X_{2i}}{X[2i]} be the \eqn{i}th row of
\eqn{X_2}{X[2]}, \eqn{\beta_1}{\beta[1]} be a coefficient vector of length
\eqn{k_1}{k[1]} and \eqn{\beta_2}{\beta[2]} be a coefficient vector of length
\eqn{k_2}{k[2]}. Define the latent response for stage one to be
\deqn{y_{1i}^\star = X_{1i} \beta_1 + \epsilon_{1i}}{y[1i]* = X[1i] \beta[1] + \epsilon[1i]}
and stage two to be
\deqn{y_{2i}^\star = X_{2i} \beta_2 + \epsilon_{2i}.}{y[2i]* = X[2i] \beta[2] + \epsilon[2i].}
Define the observed outcome to be
\eqn{y_{i}=min(y_{1i}^\star, y_{2i}^\star).}{y[i]=min(y[1i]*,y[2i]*).} The pair
 \eqn{(\epsilon_{1i},\epsilon_{2i})}{(\epsilon[1i],\epsilon[2i])} is distributed independently
 and identically multivariate normal with means
 \eqn{E[\epsilon_{1i}] = E[\epsilon_{2i}] = 0}{E(\epsilon[1i]) = E(\epsilon[2i]) = 0},
 variances
 \eqn{Var[\epsilon_{1i}] = \sigma_{11}}{Var(\epsilon[1i]) = \sigma[11]}, \eqn{Var[\epsilon_{2i}] = \sigma_{22}}{Var(\epsilon[2i]) = \sigma[22]},
 and covariance
 \eqn{Cov(\epsilon_{1i},\epsilon_{2i}) = \sigma_{12}}{Cov(\epsilon[1i],\epsilon[2i]) = \sigma[12]}.

 The model is estimated by
 (frequentist) maximum likelihood. The default maximum likelihood algorithm is based off the Limited-memory
 Broyden Fletcher Goldfarb Shanno (L-BFGS) algorithm. See
 \link[optimr]{optimr} for details.
}
\details{
Models for \code{DE} are specified symbolically. A typical
 model has the form \code{response ~ terms1 | terms2} where \code{response}
 is the name of the numeric response variable and \code{terms1} and \code{terms2}
 are each a series of variables that specifies the linear predictor(s) of the respective model.
 For example, if the first equation has two independent variables, \code{X1} and \code{X2},
 then \code{terms1 = X1 + X2}.

A \code{formula} has an implied intercept term for both equations. To remove the
 intercept from equation 1 use either \code{response ~ terms1 - 1 | terms2}
 or \code{response ~ 0 + terms1 | terms2}. The intercept may be removed from equation 2 analgously.

The \code{control} argument is a list that can supply any of the following components:

\describe{

\item{method}{A method to be used in the function \code{optimr}. The default is \code{"L-BFGS-B"}.
 See \link[optimr]{optimr} for further details.}

\item{lower, upper}{Bounds on the variables for methods such as \code{"L-BFGS-B"} that can handle box
 (or bounds) constraints. The default is \code{-Inf} and \code{Inf}, respectively.}

\item{hessian}{A logical control that if TRUE forces the computation of an approximation to the Hessian
 at the final set of parameters. See \link[optimr]{optimr} for further details. The default is FALSE.}

\item{na.action}{A function indicating what happens when data contains NAs. The default is \link{na.omit}.
 The only other possible value is \link{na.fail}.}

\item{transformR3toPD}{A logical. If TRUE, the covariance matrix will be manually converted to a
 positive definite matrix for optimization. The default is TRUE.}

\item{Equation1Name}{A string name for the first equation. The default is "_1".}

\item{Equation2Name}{A string name for the second equation. The default is "_2".}

\item{MaskRho}{A logical or numeric to determine if the correlation is masked.
A value of FALSE means the correlation is not fixed. A value between -1 and 1 will fix the
correlation to that value. The default is FALSE. A free correlation parameter can be numerically unstable, use with caution.}

}
}
\examples{
set.seed(1775)
library(MASS)
beta01 = c(1,1)
beta02 = c(-1,-1)
N = 10000
SigmaEps = diag(2)
SigmaX = diag(2)
MuX = c(0,0)
par0 = c(beta01, beta02, SigmaX[1, 1], SigmaX[1, 2], SigmaX[2, 2])

Xgen = mvrnorm(N,MuX,SigmaX)
X1 = cbind(1,Xgen[,1])
X2 = cbind(1,Xgen[,2])
X = list(X1 = X1,X2 = X2)
eps = mvrnorm(N,c(0,0),SigmaEps)
eps1 = eps[,1]
eps2 = eps[,2]
Y1 = X1 \%*\% beta01 + eps1
Y2 = X2 \%*\% beta02 + eps2
Y = pmin(Y1,Y2)
df = data.frame(Y = Y, X1 = Xgen[,1], X2 = Xgen[,2])

results = DE(formula = Y ~ X1 | X2, data = df)

str(results)

}
\references{
\cite{Gourieroux, C. (2000). Econometrics of Qualitative Dependent Variables (Themes in Modern Econometrics) (P. Klassen, Trans.). Cambridge: Cambridge University Press. http://doi.org/10.1017/CBO9780511805608}

\cite{Maddala, G. (1983). Limited-Dependent and Qualitative Variables in Econometrics (Econometric Society Monographs). Cambridge: Cambridge University Press. http://doi.org/10.1017/CBO9780511810176}
}
