% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/S_earl.R
\name{earl}
\alias{earl}
\title{Efficient Augmentation and Relaxation Learning}
\usage{
earl(
  ...,
  moPropen,
  moMain,
  moCont,
  data,
  response,
  txName,
  regime,
  iter = 0L,
  fSet = NULL,
  lambdas = 0.5,
  cvFolds = 0L,
  surrogate = "hinge",
  kernel = "linear",
  kparam = NULL,
  verbose = 2L
)
}
\arguments{
\item{...}{Used primarily to require named input. However, inputs for
the optimization methods can be sent through the ellipsis. If surrogate
is hinge, the optimization method is dfoptim::hjk(). For all other 
surrogates, stats::optim() is used.}

\item{moPropen}{An object of class modelObj or modelObjSubset, which 
defines the model and
R methods to be used to obtain parameter estimates and
predictions for the propensity for treatment.
See ?moPropen for details.}

\item{moMain}{An object of class modelObj or modelObjSubset, which 
defines the model and
R methods to be used to obtain parameter estimates and
predictions for the main effects of the outcome.
See ?modelObj for details.}

\item{moCont}{An object of class modelObj or modelObjSubset, which 
defines the model and
R methods to be used to obtain parameter estimates and
predictions for the contrasts of the outcome.
See ?modelObj for details.}

\item{data}{A data frame of the covariates and tx histories}

\item{response}{The response variable.}

\item{txName}{A character object.
The column header of \emph{data} that corresponds to the tx covariate}

\item{regime}{A formula object or a list of formula objects.
The covariates to be included in classification. If a list is provided,
this specifies that there is an underlying subset structure -- fSet must
then be defined.}

\item{iter}{Maximum number of iterations for outcome regression}

\item{fSet}{A function or NULL defining subset structure}

\item{lambdas}{A numeric object or a numeric vector object giving the
penalty tuning parameter. If more than 1 is provided,
the finite set of values to be considered in the
cross-validation algorithm}

\item{cvFolds}{If cross-validation is to be used to select the tuning
parameters, the number of folds.}

\item{surrogate}{The surrogate 0-1 loss function must be one of
logit, exp, hinge, sqhinge, huber}

\item{kernel}{A character object.
must be one of {linear, poly, radial}}

\item{kparam}{A numeric object of NULL.
If kernel = linear, kparam is ignored.
If kernel = poly, kparam is the degree of the polynomial
If kernel = radial, kparam is the inverse bandwidth of the
kernel. If a vector of bandwidth parameters is given,
cross-validation will be used to select the parameter}

\item{verbose}{An integer or logical. If 0, no screen prints are generated. If 1,
screen prints are generated with the exception of optimization results
obtained in iterative algorithm. If 2, all screen prints are generated.}
}
\value{
an EARL object
}
\description{
Efficient Augmentation and Relaxation Learning
}
\examples{

# Load and process data set
data(bmiData)

# define the negative 12 month change in BMI from baseline
y12 <- -100*(bmiData[,6L] - bmiData[,4L])/bmiData[,4L]

# propensity model
moPropen <- buildModelObj(model = ~parentBMI+month4BMI,
                          solver.method = 'glm',
                          solver.args = list('family'='binomial'),
                          predict.method = 'predict.glm',
                          predict.args = list(type='response'))

# outcome model
moMain <- buildModelObj(model = ~parentBMI+month4BMI,
                        solver.method = 'lm')

moCont <- buildModelObj(model = ~parentBMI+month4BMI,
                        solver.method = 'lm')

fitEARL <- earl(moPropen = moPropen, moMain = moMain, moCont = moCont,
              data = bmiData, response = y12,  txName = 'A2', 
              regime = ~ parentBMI + month4BMI,
              surrogate = 'logit', kernel = 'poly', kparam = 2)

##Available methods

  # Coefficients of the regression objects
  coef(fitEARL)

  # Description of method used to obtain object
  DTRstep(fitEARL)

  # Estimated value of the optimal treatment regime for training set
  estimator(fitEARL)

  # Value object returned by regression methods
  fitObject(fitEARL)

  # Summary of optimization routine
  optimObj(fitEARL)

  # Estimated optimal treatment for training data
  optTx(fitEARL)

  # Estimated optimal treatment for new data
  optTx(fitEARL, bmiData)

  # Value object returned by outcome regression method
  outcome(fitEARL)

  # Plots if defined by regression methods
  dev.new()
  par(mfrow = c(2,4))

  plot(fitEARL)
  plot(fitEARL, suppress = TRUE)

  # Value object returned by propensity score regression method
  propen(fitEARL)

  # Parameter estimates for decision function
  regimeCoef(fitEARL)

  # Show main results of method
  show(fitEARL)

  # Show summary results of method
  summary(fitEARL)
 
}
\references{
Ying-Qi Zhao, Eric Laber, Sumona Saha and Bruce E. Sands
   (2016+)
   Efficient augmentation and relaxation learning for treatment
   regimes using observational data
}
\seealso{
Other statistical methods: 
\code{\link{bowl}()},
\code{\link{iqLearn}},
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{owl}()},
\code{\link{qLearn}()},
\code{\link{rwl}()}

Other single decision point methods: 
\code{\link{optimalClass}()},
\code{\link{optimalSeq}()},
\code{\link{owl}()},
\code{\link{qLearn}()},
\code{\link{rwl}()}

Other weighted learning methods: 
\code{\link{bowl}()},
\code{\link{owl}()},
\code{\link{rwl}()}
}
\concept{single decision point methods}
\concept{statistical methods}
\concept{weighted learning methods}
