% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DoFCorrection.R
\name{DoFCorrection}
\alias{DoFCorrection}
\title{Degrees of Freedom}
\usage{
DoFCorrection(varEstA, varEstB = varEstA, varA, varB = varA,
  method = c("WS", "JR"))
}
\arguments{
\item{varEstA}{the varEstInput object returned from certain functions, such as
\code{\link{lm.sdf}} when \code{returnVarEstInputs=} \code{TRUE}).
The variable \code{varA} must be on this data.
See Examples.}

\item{varEstB}{similar to the \code{varEstA} argument.
If left blank, both are assumed to come 
from \code{varEstA}. When set, the degrees of freedom
are for a contrast between \code{varA} and \code{varB},
and the \code{varB} values are taken from \code{varEstB}.}

\item{varA}{a character that names the statistic in the \code{varEstA}
argument for which the degrees of freedom calculation is required.}

\item{varB}{a character that names the statistic in the \code{varEstB}
argument for which a covariance is required. When \code{varB} 
is specified, returns the degrees of freedom for
the contrast between \code{varA} and \code{varB}.}

\item{method}{a character that is either \code{WS} for the Welch-Satterthwaite
formula or 
\code{JR} for the Johnson-Rust correction to the
Welch-Satterthwaite formula.}
}
\value{
numeric; the estimated degrees of freedom
}
\description{
Calculates the degrees of freedom for a statistic
             (or of a contrast between two statistics) based on the
             jackknife and imputation variance estimates.
}
\details{
This calculation happens under the notion that statistics
have little variance within strata, and
some strata will contribute fewer than a full degree of freedom.

Note that these functions are not vectorized, so \code{varA} and 
\code{varB} must contain exactly one variable name.

The method used to compute the degrees of freedom is in the
\href{https://www.air.org/sites/default/files/EdSurvey-Statistics.pdf}{Statistics vignette}
section \dQuote{Estimation of Degrees of Freedom.}
}
\examples{
sdf <- readNAEP(system.file("extdata/data", "M36NT2PM.dat", package="NAEPprimer"))
lm1 <- lm.sdf(composite ~ dsex + b017451, sdf, returnVarEstInputs=TRUE)
summary(lm1)
# this output agrees with summary of lm1 coefficient for dsex
DoFCorrection(lm1$varEstInputs,
              varA="dsexFemale",
              method="JR")
# second example, a covariance term requires more work
# first, estimate the covariance between two regression coefficients
# note that the variable names are parallel to what they are called in lm1 output
covFEveryDay <- varEstToCov(lm1$varEstInputs,
                            varA="dsexFemale",
                            varB="b017451Every day",
                            jkSumMultiplier=EdSurvey:::getAttributes(sdf, "jkSumMultiplier"))
# second, find the difference and the SE of the difference
se <- lm1$coefmat["dsexFemale","se"] + lm1$coefmat["b017451Every day","se"] +
      -2*covFEveryDay
# third, calculate the t-statistic
tv <- (coef(lm1)["dsexFemale"] - coef(lm1)["b017451Every day"])/se
# fourth, calculate the p-value, which requires the estimated degrees of freedom
dofFEveryDay <- DoFCorrection(lm1$varEstInputs,
                              varA="dsexFemale",
                              varB="b017451Every day",
                              method="JR")
# finally, the p-value
2*(1-pt(abs(tv), df=dofFEveryDay))
}
\references{
Johnson, E. G., & Rust, K. F. (1992). Population inferences and variance estimation for NAEP data. \emph{Journal of Educational Statistics,} \emph{17,} 175--190.
}
\author{
Paul Bailey
}
